<?php

namespace Mtc\Auction\Drivers;

use Carbon\Carbon;
use Mtc\Auction\Auction;
use Mtc\Auction\Contracts\Drivers\DataImporter;
use Mtc\Auction\CustomField;
use Mtc\Auction\Lot;

class ImportAuctionLots implements DataImporter
{
    use DataImportable;

    protected $model = Lot::class;

    public static $base_columns = [
        [
            'attribute_name'    => 'name',
            'display_name'      => 'Lot Name',
            'match_suggestions' => ['name'],
        ],
        [
            'attribute_name'    => 'reference',
            'display_name'      => 'Lot Reference/SKU',
            'match_suggestions' => ['id', 'sku'],
        ],
        [
            'attribute_name'    => 'barcode',
            'display_name'      => 'Lot Barcode',
            'match_suggestions' => ['barcode'],
        ],
        [
            'attribute_name'    => 'description',
            'display_name'      => 'Full Description',
            'match_suggestions' => ['description'],
        ],
        [
            'attribute_name'    => 'short_description',
            'display_name'      => 'Short Description',
            'match_suggestions' => ['short_description', 'short description'],
        ],
        [
            'attribute_name'    => 'brand_id',
            'display_name'      => 'Brand ID',
            'match_suggestions' => [
                'brand',
                'brand_id',
                'brand id',
                'distillery',
                'distillery_id',
                'distillery id',
            ],
        ],
        [
            'attribute_name'    => 'auction_id',
            'display_name'      => 'Auction ID',
            'match_suggestions' => ['auction', 'categories', 'auction_id', 'auction id'],
        ],
        [
            'attribute_name'    => 'published',
            'display_name'      => 'Published',
            'match_suggestions' => ['published'],
        ],
        [
            'attribute_name'    => 'reserve_price',
            'display_name'      => 'Reserve Price',
            'match_suggestions' => ['reserve', 'reserve_price', 'reserve price'],
        ],
        [
            'attribute_name'    => 'seller_id',
            'display_name'      => 'Seller ID',
            'match_suggestions' => ['seller', 'seller id', 'seller_id'],
        ],
        [
            'attribute_name'    => 'buyer_id',
            'display_name'      => 'Buyer ID',
            'match_suggestions' => ['buyer', 'buyer id', 'buyer_id'],
        ],
        [
            'attribute_name'    => 'seller_commission_percentage',
            'display_name'      => 'Seller Commission Percentage',
            'match_suggestions' => [
                'sellers_comission',
                'sellers_commission',
                'sellers comission',
                'sellers commission',
                'sellers_comission_percentage',
                'sellers_commission_percentage',
                'sellers comission percentage',
                'sellers commission percentage',
            ],
        ],
        [
            'attribute_name'    => 'buyer_commission_percentage',
            'display_name'      => 'Buyer Commission Percentage',
            'match_suggestions' => [
                'buyers_comission',
                'buyers_commission',
                'buyers comission',
                'buyers commission',
                'buyers_comission_percentage',
                'buyers_commission_percentage',
                'buyers comission percentage',
                'buyers commission percentage',
            ],
        ],
        [
            'attribute_name'    => 'starting_price',
            'display_name'      => 'Starting Price',
            'match_suggestions' => ['starting_price', 'starting price'],
        ],
        [
            'attribute_name'    => 'weight',
            'display_name'      => 'Weight (KG)',
            'match_suggestions' => ['weight'],
        ],
        [
            'attribute_name'    => 'is_featured',
            'display_name'      => 'Featured Lot',
            'match_suggestions' => ['featured', 'featured_lot', 'featured lot', 'is_featured', 'is featured'],
        ],
        [
            'attribute_name'    => 'is_in_slideshow',
            'display_name'      => 'Lot Appears in the Slideshow',
            'match_suggestions' => ['is_in_slideshow', 'is in slideshow', 'in_slideshow', 'in slideshow', 'slideshow'],
        ],
        [
            'attribute_name'    => 'lot_condition',
            'display_name'      => 'Text explaining a lot\'s condition',
            'match_suggestions' => ['lot_condition', 'lot condition', 'condition',],
        ],
    ];

    protected $columns = [];

    protected $defaults = [
        'auction_id'    => 0,
        'seller_id'     => null,
        'buyer_id'      => null,
        'buy_order_id'  => null,
        'published'     => 1,
        'ends_at'       => 'NOW',
        'status'        => 1,
        'name'          => '',
        'description'   => '',
        'reserve_price' => 0,
        'custom_field_set_id' => 1,
    ];

    public function mapData($import_line, $attribute_name)
    {
        if ($import_line == "Don't import") {
            return '';
        }

        switch ($attribute_name) {
            case 'auction_id':
                if (substr($import_line, 0, 10) === 'Auction > ') {
                    $auction     = Auction::firstOrCreate(['name' => substr($import_line, 10)]);
                    $import_line = $auction->id;
                }

                return $import_line;
                break;

            case 'short_description':
            case 'description':
                return '<p>' . preg_replace('/\R/u', "</p>\n<p>", $import_line) . '</p>';
                break;
            case 'reference':
                $lots_with_lot_id = Lot::query()
                                       ->where('reference', $import_line)
                                       ->get();
                if ($lots_with_lot_id->count() > 0) {
                    $import_line = '';
                }

                return $import_line;
                break;
            case 'ends_at':
                if ($import_line === 'NOW') {
                    $import_line = Carbon::now();
                }

                return $import_line;
                break;
            default:
                return $import_line;
                break;
        }
    }

    public static function getColumns(){
        return array_merge(ImportAuctionLots::$base_columns, ImportAuctionLots::getCustomColumns());
    }

    public static function getCustomColumns(){
        $lot_custom_columns = CustomField::all();
        $mapped_array = [];

        foreach ($lot_custom_columns as $column){
            $mapped_array[] = [
                'attribute_name'    => 'custom.' . $column->database_field,
                'display_name'      => $column->name,
                'match_suggestions' => [$column->database_field, $column->name],
            ];
        }

        return $mapped_array;
    }
}