<?php

namespace Mtc\Auction\Events;

use Mtc\Auction\AuctionMember;
use Mtc\Auction\Bid;

class BidUpdate
{
    /**
     * Bid that was placed
     *
     * @var Bid $bid
     */
    protected  $bid;

    /**
     * BidUpdate constructor.
     *
     * @param Bid $bid
     */
    public function __construct(Bid $bid)
    {
        $bid->load([
            'member',
            'lot'
        ]);
        $this->bid = $bid;
    }

    /**
     * Format the response for the bid
     *
     * @return array
     */
    public function getData()
    {
        $winning_sums = $winning_counts = $selling_sums = $selling_counts = [];

        if (!empty($this->bid->lot)) {
            $winning_sums = $this->bid->lot->bids()
                                           ->groupBy('member_id')
                                           ->get()
                                           ->mapWithKeys(function ($bid) {
                                               $auction_member = AuctionMember::find($bid->member->id);
                                               return [$bid->member->hashed_id => $auction_member->getLotsWinningSum()];
                                           });

            $winning_counts = $this->bid->lot->bids()
                                             ->groupBy('member_id')
                                             ->get()
                                             ->mapWithKeys(function ($bid) {
                                                 $auction_member = AuctionMember::find($bid->member->id);
                                                 return [$bid->member->hashed_id => $auction_member->getLotsWinningCount()];
                                             });

            $selling_sums = $this->bid->lot->bids()
                                           ->groupBy('member_id')
                                           ->get()
                                           ->mapWithKeys(function ($bid) {
                                               $auction_member = AuctionMember::find($bid->member->id);
                                               return [$bid->member->hashed_id => $auction_member->getLotsSellingSum()];
                                           });

            $selling_counts = $this->bid->lot->bids()
                                             ->groupBy('member_id')
                                             ->get()
                                             ->mapWithKeys(function ($bid) {
                                                 $auction_member = AuctionMember::find($bid->member->id);
                                                 return [$bid->member->hashed_id => $auction_member->getLotsSellingCount()];
                                             });
        }

        return [
            'lot_id' => $this->bid->lot_id,
            'bid_value' => $this->bid->lot->latestBid->bid ?? $this->bid->lot->starting_price ?? 0,
            'owner' => $this->bid->lot->latestBid->member->hashed_id ?? null,
            'ends_at' => $this->bid->lot->ends_at ?? null,
            'reserve_met' => $this->bid->lot->reserve_met ?? null,
            'time_remaining' => $this->bid->lot->time_remaining ?? null,
            'winning_sums' => $winning_sums,
            'winning_counts' => $winning_counts,
            'selling_sums' => $selling_sums,
            'selling_counts' => $selling_counts,
        ];
    }
}