<?php

namespace Mtc\Auction;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Mtc\Auction\Http\Response\Lotlist;
use Mtc\Cms\Models\Language;
use Mtc\Auction\Auction;
use Mtc\Auction\Filter\SortByEndTime;
use Mtc\Auction\Filter\SortByLotId;
use Mtc\Auction\Filter\SortByMostRecent;
use Mtc\Modules\Auction\Classes\Filter\SortByName;
use Mtc\Auction\Filter\SortByPopularity;
use Mtc\Auction\Filter\SortByPrice;
use Mtc\Auction\Filter\SortByRelevance;

//use Mtc\Modules\Auction\Classes\Http\Response\LotList;
use Mtc\Auction\Lot;
use Mtc\Auction\Lot\Custom;
use Mtc\Auction\Lot\Status;
use Illuminate\Support\Facades\Auth;
use Mtc\Auction\CustomFieldSetField;
use Mtc\Stemmer\Stemmer;

//use function Mtc\Modules\Auction\Classes\str_plural; temporarily replaced with local function "pluralise_string"
//use function Mtc\Modules\Auction\Classes\str_slug; temporarily replaced with local function "slugify_string"

/**
 * Class Auction
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Filter
{
    /**
     * url slug for current auction
     */
    const BUY_IT_NOW_URL_SLUG = 'buy-it-now';

    /**
     * url slug for current auction
     */
    const ACTIVE_AUCTION_URL_SLUG = 'live-auction';

    /**
     * url slug for current auction
     */
    const PAST_AUCTIONS_URL_SLUG = 'past-auctions';

    /**
     * url slug for watchlist lots
     */
    const WATCHLIST_URL_SLUG = 'my-watchlist';

    /**
     * base numerator for per page
     */
    const PER_PAGE_BASE = 36;

    /**
     * Default values for sort by list
     */
    const SORT_BY_DEFAULT = 'highest-bids';

    /**
     * Filter query
     *
     * @var Builder $query
     */
    protected $query;

    /**
     *  Filters
     *
     * @var Collection $filters
     */
    protected $active_filters;
    
    
    /**
     *  Brands
     *
     * @var Collection $brands
     */
    protected $active_brands;

    /**
     *  Filters
     *
     * @var Collection $filters
     */
    public $filter_options;


    /**
     *  Brands
     *
     * @var Collection $brands
     */
    public $brand_options;

    /**
     * Items to show per page
     *
     * @var int $per_page
     */
    public $per_page = self::PER_PAGE_BASE;

    /**
     * Current sort by value
     *
     * @var string $sort_by
     */
    public $sort_by = self::SORT_BY_DEFAULT;

    /**
     * Flag for identifying if sort by is set in url
     *
     * @var bool
     */
    protected $sort_by_from_url = false;

    /**
     * @var int
     */
    public $page = 1;

    /**
     * Pagination offset
     *
     * @var int $offset
     */
    protected $offset = 0;

    /**
     * Filtering is grouped per auctions
     *
     * @var Auction $auction
     */
    public $auction;

    /**
     * The string to search for when searching via the search box in the header
     *
     * @var string
     */
    public $search = null;

    /**
     * Total number of lots in this filter
     *
     * @var int $total_count
     */
    public $total_count;

    /**
     * Filter Language
     *
     * @var string $filter_language
     */
    public $filter_language;

    /**
     * Exclude Columns
     *
     * @var array $exclude_columns
     */
    public $exclude_columns = [
        'lot_type',
    ];

    /**
     * Build up the base query
     *
     * @param string $auction_slug
     */
    public function __construct($auction_slug)
    {
        $this->decodeAuctionSlugToId($auction_slug);
        $this->query = Lot::query()
                          ->viewable()
                          ->with([
                              'custom',
                          ]);

        if ($this->auction) {
            $this->query->where('auction_id', $this->auction->id);
        }

        if ($this->auction) {
            switch ($this->auction->admin_type) {
                case 'live':
                    $this->query->live();
                    break;
                case 'past':
                    $this->query->past();
                    break;
                case 'upcoming':
                    $this->query->future();
                    break;
            }
        } elseif ($auction_slug === self::PAST_AUCTIONS_URL_SLUG) {
            $this->query->past();
        } elseif ($auction_slug === self::BUY_IT_NOW_URL_SLUG) {
            $this->query->buyItNow();
        } elseif ($auction_slug === self::WATCHLIST_URL_SLUG) {
            $this->query->whereHas('watchers', function ($bid_query) {
                return $bid_query->where('member_id', MemberAuth::user()->id);
            })
                        ->whereIn('status', Lot\Status::$watchlist_statuses);
        }
    }

    /**
     * Only get Lots with a Translation in the current language
     * Also setting the locale for later code
     * The value should be validated outside of this function using the Controller that uses it
     *
     * @param string $locale
     */
    public function setLanguage($locale)
    {
        App::setLocale($locale);

        $language = Language::where('code', $locale)
                            ->first();

        if (is_numeric($language->id)) {
            $language_id = $language->id;
            $this->query->whereHas('translations', function ($query) use ($language_id) {
                $query->where('language_id', $language_id);
            });
        }
    }

    /**
     * Get the auction ID from url slug
     *
     * @param $auction_slug
     */
    protected function decodeAuctionSlugToId($auction_slug)
    {
        if ($auction_slug == self::PAST_AUCTIONS_URL_SLUG || $auction_slug == self::BUY_IT_NOW_URL_SLUG
            || $auction_slug == self::WATCHLIST_URL_SLUG) {
            return;
        }

        if ($auction_slug == self::ACTIVE_AUCTION_URL_SLUG) {
            $this->auction = Auction::currentLiveAuction();
        } else {
            $this->auction = Auction::query()
                                    ->whereHas('node', function ($node_query) use ($auction_slug) {
                                        return $node_query->where('url', '/' . $auction_slug);
                                    })
                                    ->first();
        }
    }

    /**
     * Checks if the applied filters are auxiliary ones (i.e. sort / per page / pagination)
     *
     * @param $filters
     *
     * @return Collection
     */
    public function processAuxiliaryFilters($all_filters)
    {
        return collect($all_filters)->reject(function ($filter) {
            return $this->isPerPageFilter($filter)
                   || $this->isPageFilter($filter)
                   || $this->isSortByFilter($filter)
                   || empty($filter);
        });
    }

    /**
     * Check and apply per page filter
     *
     * @param $filter
     *
     * @return bool
     */
    protected function isPerPageFilter($filter)
    {
        if (preg_match('/^([0-9]+)-per-page$/', $filter, $matches)) {
            $this->per_page = $matches[1];

            return true;
        }

        return false;
    }

    /**
     * Check and make sure pagination is handled
     *
     * @param $filter
     *
     * @return bool
     */
    protected function isPageFilter($filter)
    {
        if (preg_match('/^page-([0-9]+)$/', $filter, $matches)) {
            $this->page   = $matches[1];
            $this->offset = ($this->page - 1) * $this->per_page;

            return true;
        }

        return false;
    }

    /**
     * Check if filter is for sorting entries
     *
     * @param $filter
     *
     * @return bool
     */
    protected function isSortByFilter($filter)
    {
        if (array_key_exists($filter, $this->sortByOptions())) {
            $this->sort_by          = $filter;
            $this->sort_by_from_url = true;

            return true;
        }

        return false;
    }

    /**
     * @param array   $filters
     * @param Request $request
     */
    public function applyFilters(Request $request, $filters = [], $brands = [], $provided_search = '')
    {        
        $this->active_filters = $filters;
        $this->active_brands = $brands;
        $this->filter_options = self::filterOptionList($this->auction->id ?? 0);
        $this->brand_options = self::filterBrandList($this->auction->id ?? 0);
        //$this->filter_options = $this->filter_options->merge(self::filterBrandList($this->auction->id ?? 0));
        $this->search         = $request->input('search_filter');
        if (empty($this->search) && !empty($provided_search)) {
            $this->search = $provided_search;
        }
        $request->wantsJson() ? $this->setJsonFilters($request) : $this->splitFiltersByType();
        $this->filterBaseQuery();

        // Default search to relevance if this is a term search
        if (!empty($this->search) && !$this->sort_by_from_url && empty($request->input('sort'))) {
            $this->sort_by = 'relevance';
        }
    }

    /**
     * @param array   $filters
     * @param Request $request
     */
    public function applyFiltersNoRequest($filters = [], $brands = [], $provided_search = '', $sortby = 'relevance')
    {
        $this->active_filters = $filters;
        $this->active_brands = $brands;
        $this->filter_options = self::filterOptionList($this->auction->id ?? 0);
        $this->brand_options = self::filterBrandList($this->auction->id ?? 0);
        //$this->filter_options = $this->filter_options->merge(self::filterBrandList($this->auction->id ?? 0));
        $this->search         = $provided_search;

        $this->splitFiltersByType();
        $this->filterBaseQuery();

        // Default search to relevance if this is a term search
        if (!empty($this->search) && !$this->sort_by_from_url && empty($sortby)) {
            $this->sort_by = 'relevance';
        }
    }

    /**
     * @param Request $request
     */
    protected function setJsonFilters(Request $request)
    {
        $this->page     = $request->input('page', 1);
        $this->per_page = $request->input('per_page', self::PER_PAGE_BASE);
        $this->sort_by  = $request->input('sort', self::SORT_BY_DEFAULT);
        $this->offset   = ($this->page - 1) * $this->per_page;

        $this->active_filters = collect($request->input('filters', []))
            ->reject(function ($selection) {
                return empty($selection);
            })
            ->map(function ($filter_selections) {
                return collect($filter_selections);
            });
        $this->active_brands = collect($request->input('brands', []))
            ->reject(function ($selection) {
                return $selection === 'false';
            })
            ->reject(function ($selection) {
                return $selection === 'undefined';
            })
            ->map(function ($filter_selections, $key) {
                return $key;
            });


        // Initialize remaining filter arrays
        $this->filter_options->each(function ($value_list, $name) {
            if (!$this->active_filters->has($name)) {
                $this->active_filters->put($name, collect([]));
            }
        });
    }

    protected static function filterBrandList(int $auction_id)
    {
        $brand_ids = Lot::query()
                        ->whereHas('auction', function ($auction_query) use ($auction_id) {
                            if ($auction_id > 0) {
                                return $auction_query->where('auction_id', $auction_id);
                            }

                            return $auction_query->where('ends_at', '<', Carbon::now());
                        })
                        ->where('brand_id', '>', 0)
                        ->pluck('brand_id');
        
        return collect([
            'brand' => \Mtc\Shop\Brand::query()
                                      ->whereIn('id', $brand_ids)
                                      ->pluck('name', 'id'),
        ]);
    }

    /**
     * Get the list of all filter options
     *
     * @param int $auction_id
     *
     * @return mixed
     */
    protected static function filterOptionList(int $auction_id)
    {
        //return Cache::remember("auction_filter_options_{$auction_id}", 15, function () use ($auction_id) {
        return CustomFieldSetField::query()
                                  ->where('lots', 1)
                                  ->where('filter', 1)/*->with([
                    'field.propertyOptions',
                ])*/
                                  ->get()
                                  ->keyBy('field.database_field')
                                  ->map(function ($field) use ($auction_id) {
                                      return Custom::query()
                                                   ->whereHas('lot', function ($lot_query) use ($auction_id) {
                                                       return $lot_query->viewable()
                                                                        ->whereHas('auction',
                                                                            function ($auction_query) use ($auction_id
                                                                            ) {
                                                                                if ($auction_id > 0) {
                                                                                    return $auction_query->where('auction_id',
                                                                                        $auction_id);
                                                                                }

                                                                                return $auction_query->where('ends_at',
                                                                                    '<', Carbon::now());
                                                                            });
                                                   })
                                                   ->distinct($field->field->database_field)
                                                   ->pluck($field->field->database_field)
                                                   ->map(function ($name) {
                                                       return trim(trim($name), "\u{2028}");
                                                   })
                                                   ->keyBy(function ($name) {
                                                       return self::slugify_string($name);
                                                   })
                                                   ->sortBy(function ($field) {
                                                       return $field;
                                                   });
                                  });
        //});
    }

    /**
     * Go from array with values to array that is keyed by the database field for this value
     */
    protected function splitFiltersByType()
    {
        $filters_grouped = collect([]);

        // Initialize filter arrays
        $this->filter_options->each(function ($value_list, $name) use ($filters_grouped) {
            $filters_grouped->put($name, collect([]));
        });

        // Set active filters based on selections
        $this->active_filters->each(function ($searchable) use ($filters_grouped) {
            // Find in the filter groups whether there is an entry with this searchable in it
            $filter_key = $this->filter_options->filter(function ($value_list, $name) use ($searchable) {
                return $value_list->has($searchable);
            })
                                               ->keys()
                                               ->first();

            if ($filter_key) {
                $filters_grouped[$filter_key]->push($searchable);
            }

            return $filter_key ?: [];
        })
                             ->sortBy(function ($field) {
                                 return $field;
                             });

        $this->active_filters = $filters_grouped;
    }

    /**
     * Apply filters to base query
     */
    protected function filterBaseQuery()
    {
        if (!empty($this->search)) {
            $this->query->where(function ($query) {
                $query->whereRaw('MATCH `name` AGAINST (? IN BOOLEAN MODE)', self::fullTextWildcards($this->search))
                      ->orWhereRaw('MATCH `description` AGAINST (? IN BOOLEAN MODE)',
                          self::fullTextWildcards($this->search))
                      ->orWhere(config('auction.lots.identifier'), $this->search);
            });
        }

        $this->query->whereHas('custom', function ($query) {
            $this->active_filters->each(function ($values, $field) use ($query) {
                if ($values->isEmpty()) {
                    return;
                }

                // We need to map the URL friendly values with DB values
                $values = $values->map(function ($url_value) use ($field) {
                    return $this->filter_options[$field][$url_value];
                })
                                 ->filter();

                // Filter field for these values
                $query->whereIn($field, $values);
            });
        });
        
        if($this->active_brands->count() > 0){
            $this->query->whereIn('brand_id', $this->active_brands);
        }
    }

    /**
     * Get the products from filtered query
     *
     * @return Collection
     */
    public function getItems()
    {
        $this->total_count = $this->query->count();

        $this->setSort();

        return (new Lotlist($this->query->take($this->per_page)
                                        ->offset($this->offset)
                                        ->get()))->handle();
    }

    /**
     * Retrieve the list of active (set) filters
     *
     * @return Collection
     */
    public function getActiveFilters()
    {
        return $this->active_filters;
    }
    /**
     * Retrieve the list of active (set) filters
     *
     * @return Collection
     */
    public function getActiveBrands()
    {
        return $this->active_brands;
    }

    /**
     * Get the full list of filter options
     *
     * @return Collection
     */
    public function getFilters()
    {
        $filter_names = $this->filterNames(true);
        if ($this->active_filters->flatten()
                                 ->isNotEmpty()) {
            return $this->onlyApplicableFilterOptions()
                        ->each(function (Collection $filter, $filter_key) use ($filter_names) {
                            if ($filter->isNotEmpty()) {
                                //$filter->prepend('All ' . $filter_names[$filter_key], '');
                            }
                        });
        }

        //print_r($this->filter_options);

        return $this->filter_options->each(function (Collection $filter, $filter_key) use ($filter_names) {
            if (in_array($filter_key, $this->exclude_columns)) {
                $this->filter_options->forget($filter_key);
            }

            //            if ($filter->isNotEmpty()) {
            //                $filter->prepend('All ' . $filter_names[$filter_key], '');
            //            }
        });
    }

    public function getBrands(){
        $brand_names = $this->brandNames(true);
        return $this->brand_options;
        return collect();
        if ($this->active_brands->flatten()
                                 ->isNotEmpty()) {
            return $this->onlyApplicableBrandOptions()
                        ->each(function (Collection $filter, $filter_key) use ($brand_names) {
                            if ($filter->isNotEmpty()) {
                                //$filter->prepend('All ' . $filter_names[$filter_key], '');
                            }
                        });
        }

        return $this->brand_options->each(function (Collection $filter, $filter_key) use ($brand_names) {
            if (in_array($filter_key, $this->exclude_columns)) {
                $this->brand_options->forget($filter_key);
            }
        });
    }

    /**
     * Get the page URL with the selected filters
     *
     * @param $auction_slug
     *
     * @return string
     */
    public function getUrlForFilters($auction_slug)
    {
        $url_parts = $this->active_filters->flatten()
                                          ->prepend($auction_slug);

        if (!empty($this->search)) {
            $url_parts = $url_parts->push('q-' . urlencode($this->search));
        }

        if ($this->per_page != Filter::PER_PAGE_BASE) {
            $url_parts = $url_parts->push($this->per_page . '-per-page');
        }

        if ($this->page != 1) {
            $url_parts = $url_parts->push('page-' . $this->page);
        }

        if ($this->sort_by !== self::SORT_BY_DEFAULT) {
            $url_parts = $url_parts->push($this->sort_by);
        }

        return '/' . $url_parts->implode('/');
    }

    /**
     * Reduce the filter options only to those that have valid entries or are part of the selections
     *
     * @return Collection
     */
    protected function onlyApplicableFilterOptions()
    {
        return $this->filter_options->map(function ($filter_values, $field) {
            $query = Custom::query()
                           ->whereHas('lot', function ($lot_query) {
                               return $lot_query->viewable()
                                                ->where(function ($query) {
                                                    if ($this->auction) {
                                                        return $query->where('auction_id', $this->auction->id);
                                                    }

                                                    return $query->whereHas('auction', function ($auction_query) {
                                                        return $auction_query->where('ends_at', '<', Carbon::now());
                                                    });
                                                });
                           })
                           ->distinct($field);

            $this->active_filters->filter(function ($filter, $filter_name) use ($field) {
                return $filter->isNotEmpty() && $filter_name !== $field;
            })
                                 ->each(function ($active_values, $active_field) use ($query) {
                                     $active_values = $active_values->map(function ($active_value) use (
                                         $active_field
                                     ) {
                                         return $this->filter_options[$active_field][$active_value];
                                     });

                                     return $query->whereIn($active_field, $active_values);
                                 });

            return $query->pluck($field)
                         ->keyBy(function ($field) {
                             return self::slugify_string($field);
                         })
                         ->sortBy(function ($field) {
                             return $field;
                         });
        })
                                    ->reject(function ($value, $key) {
                                        return in_array($key, $this->exclude_columns);
                                    });
    }
    
    /**
     * Reduce the filter options only to those that have valid entries or are part of the selections
     *
     * @return Collection
     */
    protected function onlyApplicableBrandOptions()
    {
        return $this->brand_options->map(function ($filter_values, $field) {

            $lot_query = Lot::query()->viewable()
                      ->where(function ($query) {
                          if ($this->auction) {
                              return $query->where('auction_id', $this->auction->id);
                          }

                          return $query->whereHas('auction', function ($auction_query) {
                              return $auction_query->where('ends_at', '<', Carbon::now());
                          });
                      });
            
            

            $this->active_filters->filter(function ($filter, $filter_name) use ($field) {
                return $filter->isNotEmpty() && $filter_name !== $field;
            })
                                 ->each(function ($active_values, $active_field) use ($query) {
                                     $active_values = $active_values->map(function ($active_value) use (
                                         $active_field
                                     ) {
                                         return $this->filter_options[$active_field][$active_value];
                                     });

                                     return $query->whereIn($active_field, $active_values);
                                 });
            
            
        });
    }

    /**
     * Get the field names for auction
     *
     * @return mixed
     */
    public function filterNames($multiple = false)
    {
        $remember_key = $multiple ? 'auction_filter_field_names_multiple' : 'auction_filter_field_names';

        //return Cache::remember($remember_key, 15, function () use ($multiple) {
        return CustomFieldSetField::query()
                                  ->where('lots', 1)
                                  ->where('filter', 1)/*->with([
                    'field.propertyOptions',
                ])*/
                                  ->get()
                                  ->keyBy('field.database_field')
                                  ->map(function ($field) use ($multiple) {
                                      return $multiple ? self::pluralise_string($field->field->name) : $field->field->name;
                                  });
        //});
    }

    /**
     * Get the field names for auction
     *
     * @return mixed
     */
    public function brandNames($multiple = false)
    {
        $remember_key = $multiple ? 'auction_filter_field_names_multiple' : 'auction_filter_field_names';

        return \Mtc\Shop\Brand::query()
                              ->pluck('name', 'id');
    }

    /**
     * Get the per_page option range
     *
     * @return array
     */
    public function perPageRange()
    {
        return [
            Filter::PER_PAGE_BASE     => __('auction::lots.filter.items_per_page', ['items' => Filter::PER_PAGE_BASE]),
            Filter::PER_PAGE_BASE * 2 => __('auction::lots.filter.items_per_page',
                ['items' => Filter::PER_PAGE_BASE * 2]),
            Filter::PER_PAGE_BASE * 3 => __('auction::lots.filter.items_per_page',
                ['items' => Filter::PER_PAGE_BASE * 3]),
            Filter::PER_PAGE_BASE * 4 => __('auction::lots.filter.items_per_page',
                ['items' => Filter::PER_PAGE_BASE * 4]),
            Filter::PER_PAGE_BASE * 5 => __('auction::lots.filter.items_per_page',
                ['items' => Filter::PER_PAGE_BASE * 5]),
        ];
    }

    /**
     * list of available sort options
     *
     * @return array
     */
    public function sortByOptions()
    {
        $options = [
            'highest-bids'  => [
                'name'  => __('auction::lots.filter.sort.highest_bids'),
                'query' => SortByPrice::class,
            ],
            'lowest-bids'   => [
                'name'      => __('auction::lots.filter.sort.lowest_bids'),
                'query'     => SortByPrice::class,
                'arguments' => [
                    'direction' => 'asc',
                ],
            ],
            'latest-bids'   => [
                'name'  => __('auction::lots.filter.sort.latest_bids'),
                'query' => SortByMostRecent::class,
            ],
            'most-popular'  => [
                'name'  => __('auction::lots.filter.sort.most_popular'),
                'query' => SortByPopularity::class,
            ],
            'alphabetic-az' => [
                'name'  => __('auction::lots.filter.sort.alphabetic_az'),
                'query' => SortByName::class,
            ],
            'alphabetic-za' => [
                'name'      => __('auction::lots.filter.sort.alphabetic_za'),
                'query'     => SortByName::class,
                'arguments' => [
                    'direction' => 'desc',
                ],
            ],
            'end-time'      => [
                'name'  => __('auction::lots.filter.sort.end_time'),
                'query' => SortByEndTime::class,
            ],
            'lot_id'        => [
                'name'  => __('auction::lots.filter.sort.lot_id'),
                'query' => SortByLotId::class,
            ],
        ];

        if (!empty($this->search)) {
            $options['relevance'] = [
                'name'      => __('auction::lots.filter.sort.relevance'),
                'query'     => SortByRelevance::class,
                'arguments' => [
                    'terms' => $this->search,
                ],
            ];
        }

        return $options;
    }

    /**
     * Apply sorting on filter query
     */
    protected function setSort()
    {
        $sort_options = $this->sortByOptions();

        if (array_key_exists($this->sort_by, $sort_options)) {
            $sort_class = new $sort_options[$this->sort_by]['query']($this->query);
            $arguments  = $sort_options[$this->sort_by]['arguments'] ?? [];

            $this->query = $sort_class->handle(...array_values($arguments));
        }
    }

    /**
     * Verify Auction exists
     *
     * @return bool
     */
    public function hasAuction()
    {
        return !empty($this->auction->id);
    }

    /**
     * Get the auction slug from request sent in
     *
     * @param Request $request
     * @param array   $applied_filters
     *
     * @return mixed|string
     */
    public static function getAuctionSlugFromRequest(Request $request, $applied_filters)
    {
        // Check if doing a search from the home page
        if (!empty($request->input('search-auction-type'))) {
            switch ($request->input('search-auction-type')) {
                case 'past':
                    return Filter::PAST_AUCTIONS_URL_SLUG;
                    break;
                case 'live':
                default:
                    return Filter::ACTIVE_AUCTION_URL_SLUG;
                    break;
            }
        }

        // Check if requesting a different auction than the current
        if ($request->has('auction_id')) {
            if ($request->input('auction_id') == -3) {
                return Filter::WATCHLIST_URL_SLUG;
            }

            if ($request->input('auction_id') == -2) {
                return Filter::BUY_IT_NOW_URL_SLUG;
            }


            if ($request->input('auction_id') == -1) {
                return Filter::PAST_AUCTIONS_URL_SLUG;
            }

            if ($request->input('auction_id') == 0) {
                return Filter::ACTIVE_AUCTION_URL_SLUG;
            }

            return ltrim(Auction::find($request->get('auction_id'))->url, '/');
        }

        if ($request->wantsJson()) {
            return Filter::ACTIVE_AUCTION_URL_SLUG;
        }

        //Check if the first of the filters is a language code
        $languages_codes = Language::pluck('code')
                                   ->toArray();
        if (in_array($applied_filters[0], $languages_codes)) {
            //shift the array's pointer to the next pme
            array_shift($applied_filters);
        }

        return array_shift($applied_filters);
    }

    /**
     * Replaces spaces with full text search wildcards
     *
     * @param string $term
     *
     * @return string
     */
    public static function fullTextWildcards($term)
    {
        // removing symbols used by MySQL
        $reservedSymbols   = ['-', '+', '<', '>', '@', '(', ')', '~'];
        $term              = str_replace($reservedSymbols, '', $term);
        $wildcardSymbols   = [' ',];
        $term              = str_replace($wildcardSymbols, '*', trim($term));
        $default_stopwords = [
            'a',
            'about',
            'an',
            'are',
            'as',
            'at',
            'be',
            'by',
            'com',
            'de',
            'en',
            'for',
            'from',
            'how',
            'i',
            'in',
            'is',
            'it',
            'la',
            'of',
            'on',
            'or',
            'that',
            'the',
            'this',
            'to',
            'was',
            'what',
            'when',
            'where',
            'who',
            'will',
            'with',
            'und',
            'the',
            'www',
        ];

        $words = explode(' ', $term);

        foreach ($words as $key => $word) {
            /*
             * applying + operator (required word) only big words
             * because smaller ones are not indexed by mysql
             */
            if (strlen($word) >= 2) {
                if (!in_array($word, $default_stopwords)) {
                    $words[$key] = '+' . Stemmer::stem($word) . '*';
                }
            }
        }

        return implode(' ', $words);
    }


    /**
     * Replaces spaces with full text search wildcards
     *
     * @param string $term
     *
     * @return array
     */
    static public function filterWithArrayAndLot($lot_id, $filter)
    {
        $auction_slug = $filter['auction'];

        if (!empty($filter['auction'])) {
            switch ($filter['auction']) {
                case 'past-auctions':
                    $auction_slug = Filter::PAST_AUCTIONS_URL_SLUG;
                    break;
                case 'live-auction':
                default:
                    $auction_slug = Filter::ACTIVE_AUCTION_URL_SLUG;
                    break;
            }
        }

        // Check if requesting a different auction than the current
        if ($filter['auction_id']) {
            if ($filter['auction_id'] == -3) {
                $auction_slug = Filter::WATCHLIST_URL_SLUG;
            }

            if ($filter['auction_id'] == -2) {
                $auction_slug = Filter::BUY_IT_NOW_URL_SLUG;
            }


            if ($filter['auction_id'] == -1) {
                $auction_slug = Filter::PAST_AUCTIONS_URL_SLUG;
            }

            if ($filter['auction_id'] == 0) {
                $auction_slug = Filter::ACTIVE_AUCTION_URL_SLUG;
            }
        }

        $new_filter = new Filter($auction_slug);

        $flattened_brands   = [];
        $flattened_filters   = [];
        $flattened_filters[] = $auction_slug;

        if (!empty($filter['active_filters'])) {
            foreach ($filter['active_filters'] as $active_filter) {
                $flattened_filters[] = array_values($active_filter)[0];
            }
        }
        if (!empty($filter['active_brand'])) {
            foreach ($filter['active_brand'] as $key => $brand_value) {
                if($brand_value) {
                    $flattened_brands[$key] = $key;
                }
            }
        }
        $filter_array_to_collection = collect($flattened_filters);
        $brand_array_to_collection = collect($flattened_brands);

        $new_filter->applyFiltersNoRequest($filter_array_to_collection, $brand_array_to_collection, $filter['search_filter'], $filter['sort_by']);
        $lots_in_search = $new_filter->getItems();
        $lotIndex       = $lots_in_search->search(function ($lot) use ($lot_id) {
            return $lot->lot_id == $lot_id;
        });

        $lotIterator = $lots_in_search->getIterator();

        $previous = '';
        try {
            if ($lotIndex != 0) {
                $lotIterator->seek($lotIndex - 1);
                $previous = $lotIterator->current();
            }
        } catch (\Exception $exception) {
            //nothing, there must not be a previous one
        }

        $next = '';
        try {
            if ($lotIndex + 1 != $lots_in_search->count()) {
                $lotIterator->seek($lotIndex + 1);
                $next = $lotIterator->current();
            }
        } catch (\Exception $exception) {
            //nothing, there must not be a next one
        }

        return ['previous' => $previous, 'next' => $next,];
    }

    public static function slugify_string($string)
    {
        return strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $string)));
    }

    public static function pluralise_string($string)
    {
        return $string;
    }
}
