<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Auction\Auction;
use Mtc\Auction\Lot;
use Mtc\Core\Facades\Media;

class AuctionController extends Controller
{
    const PER_PAGE = 20;
    /**
     * View Auction List
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function index(Request $request)
    {
        $this->page_meta['title'] = 'Manage Auctions';
        $auctions = Auction::query()
                           ->latest()
                           ->orderBy('id', 'desc');

        /*$pagination = new PaginationTemplate([
            'item_count' => $auctions->count(),
            'per_page' => self::PER_PAGE,
            'active_page' => request()->input('page', 1),
            'show_view_all' => false,
            'page_url' => '/admin/auctions/',
        ]);*/
        
//        $pagination = $pagination->render($twig);
        $auctions = $auctions->paginate(config('auction.admin.pagination'))->appends($request->input());

        return template('Auction/admin/auctions/index.twig', [
            'auctions' => $auctions,
            'page_meta' => $this->page_meta,
        ]);
    }



    /**
     * View Auction Create form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Auction';
        return template('Auction/admin/auctions/edit.twig', [
            'auction' => new Auction,
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Process Auction create request
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name' => 'required',
            'content' => 'present',
            'submit_deadline' => 'required|date',
            'starts_at' => 'required|date',
            'ends_at' => 'required|date',
            'image' => 'image',
            'banner_image' => 'image',
            'title' => 'present',
            'subtitle' => 'present',
        ]);

        $auction = new Auction;
        $auction->fill($validated);
        if (isset($validated['image'])) {
            $auction->image = $this->processImageUpload('image');
        }
        
        if (isset($validated['banner_image'])) {
            $auction->banner_image = $this->processImageUpload('banner_image');
        }
        
        if(!$auction->is_live){
            $auction->original_ends_at = $auction->ends_at;            
        }
        
        $auction->save();
        
        //Make sure the node is set up and correct
        $auction->regenerateNode();

        session()->flash('success', 'Auction Created');
        return redirect()->to("/admin/auctions/{$auction->id}/edit");
    }

    /**
     * View Auction Edit form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function edit($auction_id)
    {
        $image_folders = Media::getFolders('auction_images');
        $this->page_meta['title'] = 'Edit Auction Info';

        $auction = Auction::query()->findOrFail($auction_id);

        return template('Auction/admin/auctions/edit.twig', [
            'auction'               => $auction,
            'auction_stats' => $auction->getAuctionStats(),
            'page_meta'             => $this->page_meta,
            'auction_image_folders' => $image_folders,
        ]);
    }

    /**
     * Process Auction update request
     *
     * @param Request $request
     * @param $auction_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $auction_id)
    {
        $auction = Auction::query()->find($auction_id);

        $validated = $this->validate($request, [
            'name' => 'required',
            'content' => 'present',
            'submit_deadline' => 'required|date',
            'starts_at' => 'required|date',
            'ends_at' => 'required|date',
            'image' => 'image',
            'banner_image' => 'image',
            'title' => 'present',
            'subtitle' => 'present',
        ]);
        
        $auction->fill($validated);
        if ($request->input('delete_image', false)) {
            $auction->image = '';
        }
        
        if ($request->input('delete_banner_image', false)) {
            $auction->banner_image = '';
        }
        if (isset($validated['image'])) {
            $auction->image = $this->processImageUpload('image');
        }
        
        if (isset($validated['banner_image'])) {
            $auction->banner_image = $this->processImageUpload('banner_image');
        }
        
        if(!$auction->is_live){
            $auction->original_ends_at = $auction->ends_at; 
        }
        
        
        if ($auction->isDirty('ends_at')) {
            //In the past, we would change all lots in the auction based on whether the lot ended in the past
            //Now it just updates all lots that are not live
            $auction->lots()
                    ->whereIn('status', Lot\Status::$pre_auction_statuses)
                //->where('ends_at', '<=', $auction->getOriginal('ends_at'))
                    ->update([
                                 'ends_at'          => $auction->ends_at,
                                 'original_ends_at' => $auction->original_ends_at,
                             ]);
        }
        $auction->save();
        //$auction->regenerateNode();

        session()->flash('success', 'Auction Updated');
        return redirect()->to("/admin/auctions/{$auction->id}/edit");
    }

    /**
     * Delete an auction
     *
     * @param $auction_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($auction_id)
    {
        $auction = Auction::query()->find($auction_id);
        if ($auction) {
            $auction->delete();
            session()->flash('success', 'Auction Removed');
        }
        return redirect()->back();
    }

    protected function processImageUpload($image_name)
    {
        try {
            $image_folders = Media::getFolders('auction_images');

            if ($file_data = upload_image($image_name, $image_folders)) {
                if (!empty($file_data['error'])) {
                    session()->flash('error', $file_data['error']);
                    return $this->{$image_name};
                }

                return $file_data['name'];
            }
            session()->flash('error', 'Failed to upload file. Please try again');

        } catch (\Exception $exception) {
            session()->flash('error', 'File upload failed, please make sure you upload a supported file and try again');
        }

        return $this->{$image_name};
    }
}