<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use App\Http\Controllers\Controller;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use Mtc\Auction\Auction;
use Mtc\Auction\AuctionMemberFields;
use Mtc\Auction\Lot;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Members\Http\Requests\AdminMemberRequest;
use Mtc\Members\Http\Requests\SaveAdminMemberRequest;
use Mtc\Members\Role;
use Mtc\Auction\AuctionMember;

class AuctionMemberController extends Controller
{
    /**
     * List records
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(AdminMemberRequest $request)
    {
        $this->page_meta['title'] = 'Manage Members';
        $members = $request->member()
                           ->newQuery()
                           ->search($request)
                           ->latest('id')
                           ->paginate(50)
                           ->appends($request->input());

        return template('members/admin/index.twig', [
            'page_meta' => $this->page_meta,
            'members' => $members,
        ]);
    }

    /**
     * Create new record
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(AdminMemberRequest $request)
    {
        $this->page_meta['title'] = 'Create Member Record';
        return template('Auction/admin/members/edit.twig', [
            'page_meta' => $this->page_meta,
            'member' => $request->member(),
            'roles' => Role::query()->pluck('name', 'id'),
        ]);
    }

    /**
     * Store record
     *
     * @param SaveAdminMemberRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(SaveAdminMemberRequest $request)
    {
        $request->member()->fill($request->input())->save();
        if ($request->has('role')) {
            $request->member()->syncRoles($request->input('role'));
        }
        
        $auction_settings = AuctionMemberFields::query()->where('member_id', $request->member()->id)->first();
        if(is_null($auction_settings)) {

            $auction_settings  = new AuctionMemberFields();
            $auction_settings->member_id = $request->member()->id;
            $auction_settings->generateUniqueID();
            $auction_settings->save();

        }
        
        if(empty($auction_settings->unique_id)){
            $auction_settings->generateUniqueID();
            $auction_settings->save();
        }
        
        if($request->input('buyer_membership_fee_paid')){
            $auction_settings->buyer_membership_fee_paid = true;
        } else {
            $auction_settings->buyer_membership_fee_paid = false;
        }
        
        if($request->input('seller_membership_fee_paid')){
            $auction_settings->seller_membership_fee_paid = true;
        } else {
            $auction_settings->seller_membership_fee_paid = false;
        }

        $auction_settings->seller_information = $request->input('seller_information');
        $auction_settings->save();
        
        
        return Redirect::to(route('admin.members.edit', $request->member()->id))
                       ->with('success', 'Member Saved');
    }

    /**
     * Edit a record
     *
     * @param AdminMemberRequest $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(AdminMemberRequest $request)
    {        
        $this->page_meta['title'] = 'Manage Member Record - ' . $request->member()->name;
        
        return template('Auction/admin/members/edit.twig', [
            'page_meta' => $this->page_meta,
            'member' => $request->member(),
            'auction_fields' => AuctionMemberFields::query()->where('member_id', $request->member()->id)->first(),
            'roles' => Role::query()->pluck('name', 'id'),
        ]);
    }

    /**
     * Update members details
     *
     * @param SaveAdminMemberRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(SaveAdminMemberRequest $request)
    {
        $request->member()->fill($request->input())->save();
        if ($request->has('role')) {
            $request->member()->syncRoles($request->input('role'));
        }

        $auction_settings = AuctionMemberFields::query()->where('member_id', $request->member()->id)->first();
        if(is_null($auction_settings)) {
            $auction_settings  = new AuctionMemberFields();
            $auction_settings->member_id = $request->member()->id;
            $auction_settings->generateUniqueID();
            $auction_settings->save();
        }
        
        if(empty($auction_settings->unique_id)){
            $auction_settings->generateUniqueID();
            $auction_settings->save();
        }
        
        if($request->input('buyer_membership_fee_paid')){
            $auction_settings->buyer_membership_fee_paid = true;
        } else {
            $auction_settings->buyer_membership_fee_paid = false;
        }

        if($request->input('seller_membership_fee_paid')){
            $auction_settings->seller_membership_fee_paid = true;
        } else {
            $auction_settings->seller_membership_fee_paid = false;
        }

        $auction_settings->seller_information = $request->input('seller_information');
        $auction_settings->save();
        
        return Redirect::back()->with('success', 'Member Updated');
    }

    /**
     * Remove record
     *
     * @param $member_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(AdminMemberRequest $request)
    {
        $request->member()->delete();
        return Redirect::back()->with('success', 'Member Record Removed');
    }

    /**
     * Login as member
     *
     */
    public function loginAs(AdminMemberRequest $request)
    {
        MemberAuth::login($request->member());
        return redirect(route('members.dashboard'));
    }
}