<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Auction\CollectionPoint;


class LotCollectionPointController extends Controller
{
    /**
     * View Lot Collection Point list
     *
     * @param Request $request
     * @return string
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function index(Request $request)
    {
        $this->page_meta['title'] = 'Collection Points for ' . config('app.name');
        return template('Auction/admin/lot_collection_points/index.twig', [
            'page_meta' => $this->page_meta,
            'collection_points' => CollectionPoint::query()->latest()->get(),
        ]);
    }

    /**
     * View Lot Collection Create form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Lot Collection Point';
        $collection_point = new CollectionPoint;
        $collection_point->active = true;

        return template('Auction/admin/lot_collection_points/edit.twig', [
            'collection_point' => $collection_point,
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Process Lot Collection create request
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name' => 'required',
        ]);

        $collection_point = new CollectionPoint();
        $collection_point->fill($validated);
        $collection_point->active = $request->has('active');
        $collection_point->save();

        session()->flash('success', 'Collection Point Created');
        return redirect()->to("/admin/lot-collection-points/{$collection_point->id}/edit");
    }

    /**
     * View Lot Collection Edit form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function edit($collection_point_id)
    {
        $this->page_meta['title'] = 'Edit Lot Collection Points';
        return template('Auction/admin/lot_collection_points/edit.twig', [
            'collection_point' => CollectionPoint::query()->findOrFail($collection_point_id),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Process Lot Collection update request
     *
     * @param Request $request
     * @param $collection_point_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $collection_point_id)
    {
        $collection_point = CollectionPoint::query()->find($collection_point_id);

        $validated = $this->validate($request, [
            'name' => 'required',
        ]);

        $collection_point->fill($validated);
        $collection_point->active = $request->has('active');
        $collection_point->save();

        session()->flash('success', 'Collection Point Updated');
        return redirect()->to("/admin/lot-collection-points/{$collection_point->id}/edit");
    }

    /**
     * Delete a Lot Collection
     *
     * @param $collection_point_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($collection_point_id)
    {
        $auction = CollectionPoint::query()->find($collection_point_id);
        if ($auction) {
            $auction->delete();
            session()->flash('success', 'Collection Point Removed');
        }
        return redirect()->back();
    }
}