<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use _PHPStan_a3459023a\Composer\XdebugHandler\Status;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Auction\Auction;
use Mtc\Auction\AuctionMember;
use Mtc\Auction\CustomFieldSet;
use Mtc\Auction\Http\Response\AdminLotList;
use Mtc\Auction\Lot;
use Mtc\Auction\PriceTrend;
use Mtc\Foundation\Country;
use Mtc\Members\Member;
use Mtc\Shop\Brand;
use Mtc\Auction\Lot\FeeTypes;
use Mtc\Auction\Lot\Fee;

class LotController extends Controller
{
    use FilterOverrides;

    /**
     * Processing actions that are allowed
     *
     * @var array
     */
    public static $lot_actions = [
        'set_costs'                      => 'Set Costs',
        'set_featured'                   => 'Set As Featured',
        'remove_featured'                => 'Remove from Featured',
        'set_slideshow'                  => 'Set As In Auction Slideshow',
        'remove_slideshow'               => 'Remove from Auction Slideshow',
        'delete_lots'                    => 'Delete Selected Lots',
        'invite_seller_to_pay'           => 'Invite Seller To Pay (Only Not Paid Lots)',
        'invite_buyer_to_pay'            => 'Send Reminder To Buyer (Only Sold Lots)',
        'change_status'                  => 'Change Status',
        'assign_to_next_auction'         => 'Assign Nearest Applicable Auction (Only Unassigned)',
        'assign_auction'                 => 'Assign to Auction',
        'mark_as_paid'                   => 'Mark Lots as Paid (Only Sold Lots)',
        'mark_paid_out'                  => 'Mark Lots as Paid Out to sellers',
        //'pay_out_with_transferwise'                        => 'Pay Out to sellers via Transferwise',
        'mark_as_shipped'                => 'Mark Lots as Shipped (Only Sold/Paid Lots)',
        'mark_as_cancelled'              => 'Mark Lots as Cancelled (Will ignore Paid/Shipped)',
        'clone'                          => 'Create a clone',
        'reslist'                        => 'Re-list this lot',
        'delete'                         => 'Delete these lots',
        'publish'                        => 'Publish these lots',
        'unpublish'                      => 'Unpublish these lots',
        'mark_for_relist'                        => 'Enable Automatic Relisting for these lots',
        'forbid_for_relist'                      => 'Disable Automatic Relisting for these lots',
        'mark_as_seller_receipt_as_sent' => 'Mark Seller Receipt as Sent',
        'print_all_seller_receipt'       => 'Print These Seller Receipts',
    ];

    /**
     * List of attributes that can be overwritten through get request
     *
     * @var array
     */
    public static $override_filters = [
        'seller_id',
        'buyer_id',
        'status',
        'auction_id',
    ];

    /**
     * View Lot list
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function index(Request $request)
    {
        /*if ($this->hasFilterOverrides($request)) {
            return $this->processRequestOverrides($request);
        }*/

        $this->page_meta['title'] = 'Manage Lots';

        return template('Auction/admin/lots/index.twig', [
            'page_meta'          => $this->page_meta,
            'lot_statuses'       => Lot\Status::$lot_statuses,
            'lot_auctions'       => Auction::idNameList(),
            'action_list'        => self::$lot_actions,
            'filter_choices'     => $this->getFilterChoices($request),
            'filter_overrides'   => $this->getOverrides(),
            'formatted_auctions' => Auction::idNameList()
                                           ->map(function ($value, $key) {
                                               return ['key' => $key, 'value' => $value];
                                           })
                                           ->prepend(['key' => '0', 'value' => 'None'])
                                           ->prepend(['key' => '-1', 'value' => 'All'])
                                           ->toArray(),
            'members'            => Member::query()
                                          ->get()
                                          ->keyBy('id')
                                          ->map(function ($member) {
                                              return $member->name . ' [ ' . $member->email . ' ]';
                                          }),
        ]);
    }

    public function show($lot_id)
    {
        return json_encode([
            'lot' => Lot::query()
                        ->with('custom')
                        ->findOrFail($lot_id),
        ]);
    }

    public function destroy($lot_id)
    {
        $lot = Lot::query()
                  ->find($lot_id);
        if ($lot) {
            $lot->fees()
                ->delete();
            $lot->custom()
                ->delete();
            $lot->images()
                ->delete();
            $lot->delete();
        }
        session()->flash('success', 'Lot Removed');

        return redirect()->back();
    }

    /**
     * View Lot Create form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $lot                      = new Lot;
        $this->page_meta['title'] = 'Create Lot';
        $auction_list             = Auction::idNameList();

        $price_trends = PriceTrend::idNameList();
        
        $brand_list = Brand::query()
                          ->pluck('name', 'id');

        /**
         *  Todo - Fee creation
         */

        /**
         *  Todo - Price Trends?
         */

        $barcode_lookup      = config('auction.settings.barcode_look_up') ? 'true' : 'false';
        $name_lookup_enabled = config('auction.settings.name_lookup_enabled') ? 'true' : 'false';

        return template('Auction/admin/lots/edit.twig', [
            'lot'               => $lot,
            'page_meta'         => $this->page_meta,
            'auction_list'      => $auction_list,
            'price_trends'      => $price_trends,
            'brand_list'      => $brand_list,
            'members'           => AuctionMember::sellerList(),
            'custom_field_sets' => json_encode(CustomFieldSet::idNameList()),
            'lot_statuses'      => json_encode(Lot\Status::$lot_statuses),
            'vue_form_settings' => '{
            "barcode_lookup_enabled": "'
                                   . $barcode_lookup
                                   . '", 
            "name_lookup_enabled": "'
                                   . $name_lookup_enabled
                                   . '",
            "default_weight": "'
                                   . config('auction.settings.default_weight')
                                   . '",
            "default_late_payment_fee_percentage": "'
                                   . config('auction.settings.default_late_payment_fee_percentage')
                                   . '",
            "default_seller_commission_limit": "'
                                   . config('auction.settings.default_seller_commission_limit')
                                   . '",
            "default_buyer_commission_percentage": "'
                                   . config('auction.settings.default_buyer_commission_percentage')
                                   . '",
            "default_seller_commission_percentage": "'
                                   . config('auction.settings.default_seller_commission_percentage')
                                   . '",
            }',
        ]);
    }

    /**
     * Process lot create
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name'                         => 'required',
            'auction_id'                   => 'sometimes|nullable|numeric',
            'buyer_id'                     => 'sometimes|nullable|numeric',
            'seller_id'                    => 'sometimes|nullable|numeric',
            'brand_id'                    => 'sometimes|nullable|numeric',
            'status'                       => 'sometimes|nullable|numeric',
            'price_trends_id'              => 'sometimes|nullable|numeric',
            'description'                  => 'present',
            'short_description'            => 'present',
            'other_information'            => 'present',
            'buyer_commission_percentage'  => 'numeric',
            'seller_commission_percentage' => 'numeric',
            //            'seller_commission'            => 'numeric',
            'seller_commission_limit'      => 'numeric',
            'weight'                       => 'numeric',
            'reserve_price'                => 'sometimes|nullable',
            'starting_price'               => 'sometimes|nullable',
            'custom_field_set_id'          => 'present',
            'custom'                       => 'present',
            'relisting_enabled'            => 'sometimes|nullable|numeric',
            'vat_on_hammer_price'          => 'sometimes|nullable|numeric',
            'ignore_late_payment_fee'      => 'sometimes|nullable|numeric',
            'is_in_slideshow'              => 'sometimes|nullable|numeric',
            'is_featured'                  => 'sometimes|nullable|numeric',
            'lot_condition'                => 'nullable',
        ]);

        $validated['published'] = 1;
        $lot                    = new Lot;
        $lot->fill($validated);
        if($lot->auction_id > 0){
            $lot->original_ends_at = $lot->auction->original_ends_at;
            $lot->ends_at = $lot->auction->ends_at;
        }        
        if (is_null($validated['status'])) {
            $lot->status = Lot\Status::REGISTERED;
        }
        //$lot->payout_details = $lot->seller->seller_details;
        $lot->save();

        if (!empty($validated['custom'])) {
            $lot->custom()
                ->create($validated['custom']);
        } else {
            //create empty relation
            $lot->custom()
                ->create();
        }

//        $default_lot_registration_fee = new Fee();
//        $default_lot_registration_fee->lot_id = $lot->id;
//        $default_lot_registration_fee->amount = config('auction.fees.registration.default_value');
//        $default_lot_registration_fee->description = config('auction.fees.registration.label');
//        $default_lot_registration_fee->PLU = config('auction.fees.registration.key');
//        $default_lot_registration_fee->save();
//
//        $default_lot_reserve_fee = new Fee();
//        $default_lot_reserve_fee->lot_id = $lot->id;
//        $default_lot_reserve_fee->amount = config('auction.fees.reserve.default_value');
//        $default_lot_reserve_fee->description = config('auction.fees.reserve.label');
//        $default_lot_reserve_fee->PLU = config('auction.fees.reserve.key');
//        $default_lot_reserve_fee->save();


        if ($request->wantsJson()) {
            return [
                'success' => true,
                'message' => 'Lot Created',
                'lot'     => $lot,
            ];
        } else {
            session()->flash('success', 'Lot Created');

            return redirect()->to("/admin/lots/{$lot->id}/edit");
        }
    }

    /**
     * View Lot Edit form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function edit($lot_id)
    {
        $this->page_meta['title'] = 'Edit Lot Info';
        $auction_list             = Auction::idNameList();

        $price_trends = PriceTrend::idNameList();

        $lot_fee_types = FeeTypes::query()
            ->where('status', 1)
            ->pluck('name', 'plu')
            ->toArray();

        $lot = Lot::query()
                  ->with('custom','fees')
                  ->findOrFail($lot_id);

        $brand_list = Brand::query()
                           ->pluck('name', 'id');
        
        if($lot->custom == null){
            //recate an empty relation
            $lot->custom()->create();
        }

        $barcode_lookup      = config('auction.settings.barcode_look_up') ? 'true' : 'false';
        $name_lookup_enabled = config('auction.settings.name_lookup_enabled') ? 'true' : 'false';
        
        if(is_null($lot->seller_commission_percentage)){
            $lot->seller_commission_percentage = config('auction.settings.default_seller_commission_percentage');
        }
        if(is_null($lot->buyer_commission_percentage)){
            $lot->buyer_commission_percentage = config('auction.settings.default_buyer_commission_percentage');
        }
        
        return template('Auction/admin/lots/edit.twig', [
            'lot'               => $lot,
            'lot_id'            => $lot->id,
            'next_lot'          => $lot->next(),
            'previous_lot'      => $lot->previous(),
            'page_meta'         => $this->page_meta,
            'auction_list'      => $auction_list,
            'brand_list'      => $brand_list,
            'price_trends'      => $price_trends,
            'custom_field_sets' => json_encode(CustomFieldSet::idNameList()),
            //'custom_fields' => $lot->prepareCustomFields(),
            'members'           => AuctionMember::sellerList(),
            'lot_statuses'      => json_encode(Lot\Status::$lot_statuses),
            'countries'         => Country::query()
                                          ->pluck('name', 'id'),
            'vue_form_settings' => '{
            "barcode_lookup_enabled": "'
                                   . $barcode_lookup
                                   . '", 
            "name_lookup_enabled": "'
                                   . $name_lookup_enabled
                                   . '",
            "default_weight": "'
                                   . config('auction.settings.default_weight')
                                   . '",
            "default_late_payment_fee_percentage": "'
                                   . config('auction.settings.default_late_payment_fee_percentage')
                                   . '",
            "default_seller_commission_limit": "'
                                   . config('auction.settings.default_seller_commission_limit')
                                   . '",
            "default_buyer_commission_percentage": "'
                                   . config('auction.settings.default_buyer_commission_percentage')
                                   . '",
            "default_seller_commission_percentage": "'
                                   . config('auction.settings.default_seller_commission_percentage')
                                   . '",
            }',
            'lot_fee_types'     => $lot_fee_types,
        ]);
    }

    /**
     * Process lot update
     *
     * @param Request $request
     * @param         $lot_id
     *
     * @return \Illuminate\Http\RedirectResponse|array
     */
    public function update(Request $request, $lot_id)
    {
        $validated = $this->validate($request, [
            'barcode'                      => 'numeric',
            'name'                         => 'required',
            'auction_id'                   => 'sometimes|nullable|numeric',
            'buyer_id'                     => 'sometimes|nullable|numeric',
            'seller_id'                    => 'sometimes|nullable|numeric',
            'brand_id'                    => 'sometimes|nullable',
            'status'                       => 'sometimes|nullable|numeric',
            'price_trends_id'              => 'sometimes|nullable|numeric',
            'description'                  => 'present',
            'short_description'            => 'present',
            'other_information'            => 'present',
            'buyer_commission_percentage'  => 'numeric',
            'seller_commission_percentage' => 'numeric',
            //'seller_commission' => 'numeric',
            'seller_commission_limit'      => 'numeric',
            'weight'                       => 'numeric',
            'reserve_price'                => 'sometimes|nullable|numeric',
            'starting_price'               => 'sometimes|nullable|numeric',
            'custom_field_set_id'          => 'present',
            'custom'                       => 'present',
            'relisting_enabled'            => 'present',
            'vat_on_hammer_price'          => 'present',
            'ignore_late_payment_fee'      => 'present',
            'is_in_slideshow'              => 'present',
            'is_featured'                  => 'present',
            'lot_condition'                => 'nullable',
        ]);

        /** @var Lot $lot */
        $lot = Lot::query()
                  ->with('custom')
                  ->findOrFail($lot_id);

        $lot->fill($validated);

        if ($lot->isDirty('auction_id') && !is_null($lot->auction)) {
            $lot->original_ends_at = $lot->auction->original_ends_at;
            $lot->ends_at = $lot->auction->ends_at;
        }
        
        /*if ($lot->isDirty('seller_id')) {
            $lot->payout_details = $lot->seller->seller_details;
        }*/
        
        $lot->save();

        if (is_null($lot->custom)) {
            $lot->custom()
                ->create($validated['custom']);
        } else {
            $lot->custom->fill($validated['custom']);
            $lot->custom->save();
        }

        if (!empty($request->input('lot_fees')) && is_array($request->input('lot_fees'))) {
            foreach ($request->input('lot_fees') as $lot_fee_id => $lot_fee_amount) {
                $lot->fees()
                    ->where('id', $lot_fee_id)
                    ->update(['amount' => $lot_fee_amount]);
            }
        }

        if ($request->wantsJson()) {
            return [
                'success' => true,
                'message' => 'Lot Updated',
                'lot'     => $lot,
            ];
        } else {
            session()->flash('success', 'Lot Updated');

            return redirect()->back();
        }
    }

    /**
     * Search for lots
     *
     * @param Request $request
     *
     * @return array
     */
    public function search(Request $request, $paginate = true)
    {
        $lots = Lot::query()
                   ->with([
                       'auction',
                       'seller',
                       //                       'custom',
                   ]);

        if ($request->input('id')) {
            $lots->where(config('auction.lots.identifier'), $request->input('id'));
        }

        if (!in_array($request->input('featured'), [null, -1])) {
            $lots->where(function ($query) use ($request) {
                $query->where('is_featured', $request->input('featured'));

                if (empty($request->input('featured'))) {
                    $query->orWhere('is_featured', null);
                }
            });
        }

        if (!in_array($request->input('slideshow'), [null, -1])) {
            $lots->where(function ($query) use ($request) {
                $query->where('is_in_slideshow', $request->input('slideshow'));

                if (empty($request->input('slideshow'))) {
                    $query->orWhere('is_in_slideshow', null);
                }
            });
        }

        if (!in_array($request->input('auction_id'), [null, -1])) {
            $lots->where('auction_id', $request->input('auction_id'));
        }

        if (!is_null($request->input('name'))) {
            $lots->where('name', 'REGEXP', preg_quote($request->input('name'), '/'));
        }

        if (!in_array($request->input('status'), [null, -1])) {
            $lots->where('status', $request->input('status'));
        }

        if (!in_array($request->input('buyer_id'), [null, -1])) {
            $lots->where('buyer_id', $request->input('buyer_id') ?: null);
        }

        if (!in_array($request->input('seller_id'), [null, -1])) {
            $lots->where('seller_id', $request->input('seller_id') ?: null);
        }


        if (!empty($request->input('sorted_columns'))) {
            foreach ($request->input('sorted_columns') as $column) {
                switch ($column['column_name']) {
                    case 'featured':
                        $column_name = 'is_featured';
                        break;
                    case 'slideshow':
                        $column_name = 'is_in_slideshow';
                        break;
                    case 'auction':
                        $column_name = 'auction_id';
                        break;
                    case 'seller':
                    case 'buyer':
                    case 'pricing':
                        $column_name = '';
                        break;
                    default:
                        $column_name = $column['column_name'];
                }

                if (!empty($column_name)) {
                    $lots->orderBy($column_name, $column['sort_direction']);
                }
            }
        }


        $lots->latest();
        $filtered_count = $lots->count();

        if ($paginate) {
            $offset = ($request->input('page', 1) - 1) * $request->input('per_page', 20);
            $lots->offset($offset)
                 ->take($request->input('per_page', 20));
        }

        $lot_list = $lots->get();

        return [
            'data'           => (new AdminLotList($lot_list))->handle(),
            'num_results'    => $filtered_count,
            'filter_choices' => $this->getFilterChoices($request),
        ];
    }

    /**
     * Gets a list of filter choices.
     *
     * @param Request $request
     *
     * @return array
     */
    private function getFilterChoices(Request $request)
    {
        $auction_id = !empty($request->input('auction_id'))
                      && $request->input('auction_id') > 0 ? $request->input('auction_id') : null;

        return [
            'featured'  => [
                ['key' => '-1', 'value' => 'All'],
                ['key' => '0', 'value' => 'No'],
                ['key' => '1', 'value' => 'Yes'],
            ],
            'slideshow' => [
                ['key' => '-1', 'value' => 'All'],
                ['key' => '0', 'value' => 'No'],
                ['key' => '1', 'value' => 'Yes'],
            ],
            'auction'   => Auction::idNameList()
                                  ->map(function ($value, $key) {
                                      return ['key' => $key, 'value' => $value];
                                  })
                                  ->prepend(['key' => '0', 'value' => 'None'])
                                  ->prepend(['key' => '-1', 'value' => 'All']),
            'status'    => collect(Lot\Status::$lot_statuses)
                ->map(function ($value, $key) {
                    return ['key' => $key, 'value' => $value];
                })
                ->prepend(['key' => '-1', 'value' => 'All']),
            'seller'    => Member::whereIn('id',
                Lot::when((!is_null($auction_id) && !empty($auction_id)), function ($query) use ($auction_id) {
                    $query->where('auction_id', $auction_id);
                })
                   ->groupBy('seller_id')
                   ->pluck('seller_id'))
                                 ->get()
                                 ->map(function ($member) {
                                     return [
                                         'key'   => $member->id,
                                         'value' => $member->name . ' [ ' . $member->email . ' ]',
                                     ];
                                 })
                                 ->sortBy('value')
                                 ->prepend(['key' => '0', 'value' => 'None'])
                                 ->prepend(['key' => '-1', 'value' => 'All']),
            'buyer'     => Member::whereIn('id',
                Lot::when((!is_null($auction_id) && !empty($auction_id)), function ($query) use ($auction_id) {
                    $query->where('auction_id', $auction_id);
                })
                   ->groupBy('buyer_id')
                   ->pluck('buyer_id'))
                                 ->get()
                                 ->map(function ($member) {
                                     return [
                                         'key'   => $member->id,
                                         'value' => $member->name . ' [ ' . $member->email . ' ]',
                                     ];
                                 })
                                 ->sortBy('value')
                                 ->prepend(['key' => '0', 'value' => 'None'])
                                 ->prepend(['key' => '-1', 'value' => 'All']),
        ];
    }

    public function recalculateValues(Request $request)
    {
        $vat_rate = config('VAT_RATE') ?? 20;
        // Seperating the Hammer Price from the Highest Bid
        $hammer_price          = $request->input('hammer_price');
        $hammer_price_with_vat = $hammer_price * (1 + $vat_rate * 0.01);

        $buyer_commission_percentage  = $request->input('buyer_commission_percentage');
        $seller_commission_percentage = $request->input('seller_commission_percentage');

        $buyer_commission          = $hammer_price * $buyer_commission_percentage * 0.01;
        $buyer_commission_with_vat = $buyer_commission * (1 + $vat_rate * 0.01);

        $seller_commission          = $hammer_price * $seller_commission_percentage * 0.01;
        $seller_commission_with_vat = $seller_commission * (1 + $vat_rate * 0.01);

        return [
            'hammer_price'               => $hammer_price,
            'hammer_price_with_vat'      => $hammer_price_with_vat,
            'buyer_commission'           => $buyer_commission,
            'buyer_commission_with_vat'  => $buyer_commission_with_vat,
            'seller_commission'          => $seller_commission,
            'seller_commission_with_vat' => $seller_commission_with_vat,
        ];
    }


    public function bulkActions(Request $request)
    {
        if (empty($request->input('lots'))) {
            session()->flash('error', 'No Lots selected');

            return redirect()->back();
        }
        
        switch ($request->input('action')) {
            case 'set_featured':

                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) {
                       $lot->is_featured = 1;
                       $lot->save();
                   });

                session()->flash('success', 'Lots Marked as Featured');

                return redirect()->back();
                break;
            case 'remove_featured':
                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) {
                       $lot->is_featured = 0;
                       $lot->save();
                   });

                session()->flash('success', 'Lots removed from Featured');

                return redirect()->back();
                break;
            case 'set_slideshow':
                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) {
                       $lot->is_in_slideshow = 1;
                       $lot->save();
                   });

                session()->flash('success', 'Lots marked as in the Slideshow');

                return redirect()->back();
                break;
            case 'remove_slideshow':
                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) {
                       $lot->is_in_slideshow = 0;
                       $lot->save();
                   });

                session()->flash('success', 'Lots removed from the Slideshow');

                return redirect()->back();
                break;
            case 'invite_seller_to_pay':
                session()->flash('error', 'invite_seller_to_pay needs done');
                Lot::query()
                   ->where('status', Lot\Status::NOT_PAID)
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->groupBy('seller_id')
                   ->each(function ($lot_collection) {

                       $member = $lot_collection->first()->seller;

                       $subject = __('seller.sell.lots_registered_for_you', ['site' => config('app.name')]);
                       $email = $this->twig->render('Auction/emails/seller_payment_invite.twig', [
                           'lots' => $lot_collection,
                           'member' => $member,
                       ]);

                       email($member->email, $subject, $email);
                   });
                session()->flash('success', 'Seller Payment Request Notification(s) sent');

                return redirect()->back();
                break;
            case 'invite_buyer_to_pay':
                Lot::query()
                   ->where('status', Lot\Status::SOLD)
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->groupBy('buyer_id')
                   ->each(function ($lot_collection) {

                       $member = $lot_collection->first()->buyer;

                       $subject = __('lots.purchase.buyer_payment_reminder_email_subject', ['site' => config('app.name')]);
                       $email = $this->twig->render('Auction/emails/buyer_payment_reminder.twig', [
                           'lots' => $lot_collection,
                           'member' => $member,
                           'bank_transfer_details' => config('auction.bank_transfer_details'),
                           'purchase_url' => config('app.url') . '/purchase-lots',
                       ]);

                       email($member->email, $subject, $email);


                   });

                session()->flash('success', 'Buyer Payment Request Notification(s) sent');
                return redirect()->back();
                break;
            case 'mark_as_paid':
                Lot::query()
                   ->where('status', Lot\Status::SOLD)
                   ->whereIn('id', $request->input('lots'))
                   ->update([
                       'status' => Lot\Status::PAID
                   ]);

                session()->flash('success', 'Lot(s) have been marked as Paid');
                return redirect()->back();
                break;
            case 'mark_paid_out':
                $already_paid_out = Lot::query()
                                       ->whereHas('seller_payout', function($query){
                                           $query->whereNotNull('payout_reference');
                                       })
                                       ->whereIn('id', $request->input('lots'))
                                       ->exists();
                if ($already_paid_out) {
                    session()->flash('error', 'Some lots already marked as paid out. No changes were made.');
                    return redirect()->back();
                }

                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->update([
                       'payout_made_at' => Carbon::now(),
                       'payout_reference' => $request->input('payout_reference'),
                   ]);

                $lots = Lot::query()
                           ->whereIn('id', $request->input('lots'))
                           ->get();

                if ($lots->count() > 0) {
                    $seller_member = $lots->first()->seller;

                    $subject = __('seller.sell.you_have_been_paid');
                    $email = $this->twig->render('Auction/emails/paid_seller.twig', [
                        'lots' => $lots,
                        'member' => $seller_member,
                        'seller_receipt_url' => SITE_URL . '/members/my-lots/' . $lots->first()->lot_id . '/receipt',
                    ]);
                    email($seller_member->email, $subject, $email);
                }

                session()->flash('success', 'Lots have been marked as Paid Out and the seller has been notified');
                return redirect()->back();
                break;
            case 'pay_out_with_transferwise':
                //On Hold until Needed

                session()->flash('error', 'Transferwise has not been set up. You should not be able to trigger this');
                return redirect()->back();
                
                $already_paid_out = Lot::query()
                                       ->whereHas('seller_payout', function($query){
                                           $query->whereNotNull('payout_reference');
                                       })
                                       ->whereIn('id', $request->input('lots'))
                                       ->exists();
                if ($already_paid_out) {
                    session()->flash('error', 'Some lots already marked as paid out. No changes were made.');
                    return redirect()->back();
                }
                if (empty(config('app.transferwise_api_token'))) {
                    session()->flash('error', 'TransferWise not set up!');
                    return redirect()->back();
                }

                $lots = collect($request->input('lots'));
                $seller_member = Lot::find($lots->first())->seller;

                if (gettype($seller_member) === 'object' && get_class($seller_member) === AuctionMember::class) {
                    
                }
                break;
            case 'assign_to_next_auction':

                $next_auction = Auction::nextAssignableAuction();
                
                if(is_null($next_auction)){
                    session()->flash('error', 'There is no assignable Auction for these lots to go to');

                } else {
                    Lot::query()
                       ->whereIn('id', $request->input('lots'))
                       ->doesntHave('auction')
                       ->get()
                       ->each(function (Lot $lot) use ($next_auction) {
                           $lot->auction_id = $next_auction->id;
                           $lot->save();
                       });
                    session()->flash('success', 'All unassigned  lots have been assigned to ' . $next_auction->name);

                }

                return redirect()->back();
                break;
            case 'assign_auction':
                $auction = Auction::query()->find($request->input('new_auction'));
                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) use ($request, $auction) {
                       $lot->auction_id = $request->input('new_auction');
                       //If the auction is not live or if the auction is live and the lot is ending in the future, update the lots end times
                       if ($auction->is_live == 0
                           || ($auction->is_live == 1
                               && $lot->ends_at->gt(
                                   $auction->original_ends_at
                               ))) {
                           $lot->ends_at = $auction->ends_at;
                           $lot->original_ends_at = $auction->original_ends_at;
                       }
                       $lot->save();
                   });
                session()->flash('success', 'Lot(s) Auction Changed');

                return redirect()->back();
                break;
            case 'delete_lots':
                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) {
                       $lot->fees()
                           ->delete();
                       $lot->custom()
                           ->delete();
                       $lot->images()
                           ->delete();
                       $lot->delete();
                   });
                session()->flash('success', 'Lots Deleted');

                return redirect()->back();
                break;
            case 'change_status':
                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) use ($request) {
                       $lot->status = $request->input('new_status');
                       $lot->save();
                   });
                session()->flash('success', 'Lots Status Changed');

                return redirect()->back();
                break;
            case 'mark_as_cancelled':
                Lot::query()
                   ->whereIn('id', $request->input('lots'))
                   ->get()
                   ->each(function (Lot $lot) {
                       $lot->status = Lot\Status::CANCELLED;
                       $lot->save();
                   });
                session()->flash('success', 'Lots Cancelled');

                return redirect()->back();
                break;
            case 'clone':
                foreach ($request->input('lots') as $lot_id) {
                    $lot = Lot::query()
                              ->find($lot_id);
                    $lot->cloneToRelist(false, true, false, true, true);
                }
                session()->flash('success', 'Lots Cloned');

                return redirect()->back();
                break;
            case 'relist':
                foreach ($request->input('lots') as $lot_id) {
                    $lot         = Lot::query()
                                      ->find($lot_id);
                    $lot->status = Lot\Status::RELISTED;
                    $lot->save();
                    $lot->cloneToRelist(false, true, true, true, false);
                }
                session()->flash('success', 'Lots Relisted');

                return redirect()->back();
                break;
            case 'mark_for_relist':
                foreach ($request->input('lots') as $lot_id) {
                    $lot         = Lot::query()
                                      ->find($lot_id);
                    $lot->relisting_enabled = 1;
                    $lot->save();
                }
                session()->flash('success', 'Lots allowed to relist automatically');

                return redirect()->back();
                break;
            case 'forbid_for_relist':
                foreach ($request->input('lots') as $lot_id) {
                    $lot         = Lot::query()
                                      ->find($lot_id);
                    $lot->relisting_enabled = 0;
                    $lot->save();
                }
                session()->flash('success', 'Lots disallowed to relist automatically');

                return redirect()->back();
                break;
            default:
                session()->flash('error', 'No Action Set');

                return redirect()->back();
        }

        session()->flash('success', 'Action Executed');

        return redirect()->back();
    }

    public function createFee($lot_id, Request $request)
    {

        $lot = Lot::query()->find($lot_id);
        if(\Mtc\Auction\Lot\Fee::feeExist($lot_id,$request->input('fee'))){

            session()->flash('error', 'Fee exist');
            return redirect()->back();

        }
        if ($lot) {
            $fee_item = FeeTypes::query()
                ->where('status', 1)
                ->where('plu', $request->input('fee'))
                ->first();

            $lot->fees()
                ->create([
                    'amount'      => $request->input('amount'),
                    'description' => $fee_item->name,
                    'PLU'         => $request->input('fee'),
                ]);
            session()->flash('success', 'Fee created');
        } else {
            session()->flash('error', 'Lot not found');
        }
        return redirect()->back();

    }

    public function destroyFee($lot_id, $fee_id)
    {
        $fee = Lot\Fee::query()->find($fee_id);
        if ($fee) {
            $fee->delete();
            session()->flash('success', 'Fee Removed');
        }
        return redirect()->back();
    }

    public function updateFee(Request $request)
    {

        if (!empty($request->input('lot_fees'))
        && is_array($request->input('lot_fees'))
        ) {
            foreach ($request->input('lot_fees') as $lot_fee_id => $lot_fee_amount) {

                $fee = Fee::find($lot_fee_id);

                if ($fee != null) {

                    Fee::where('id', $fee->id)
                        ->update(
                            [
                              'amount'=> $lot_fee_amount
                            ]
                        );
                }
                else {
                    session()->flash('error', 'Fee not found');
                    return redirect()->back();
                }

            }
        }

        session()->flash('success', 'Fee Updated');
        return redirect()->back();

    }

}