<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Mtc\Auction\Lot;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Core\Facades\Media;
use Mtc\Core\Images\ImageUploader;

class LotImageController extends Controller
{
    /**
     * View Lot Image list
     *
     * @return string
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function index(Lot $lot)
    {
        $folders = Media::getFolders('lot_images');
        
        $this->page_meta['title'] = 'Lot Images for #' . $lot->id . ': ' . $lot->name;
        return template('Auction/admin/lots/images.twig', [
            'lot' => $lot,
            'size_large' => $folders['large'],
            'zoom_large' => $folders['verylarge'],
            'next_lot' => $lot->next(),
            'previous_lot' => $lot->previous(),
            'page_meta' => $this->page_meta,
        ]);
    }



    public function show(Lot $lot)
    {
        return $lot->images()->get()
                       ->each(fn (Lot\Image $image) => $image->thumb_url = $image->getUrl('mini'));
    }
    

    /**
     * Upload Lot image
     *
     * @param Request $request
     * @param int $lot_id
     * @return string
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function store(Request $request, Lot $lot, ImageUploader $uploader)
    {
        
        $this->validate($request, [
            'image' => [
                'required',
                'image',
            ]
        ]);

        $uploader->setFolders(Media::getFolders('lot_images'));
        
        $result = $uploader->uploadImage($request->file('image'));
        if (!empty($result['name'])) {
            $image_array = [
                'name' => $result['name'],
                ];
            if($lot->images->count() === 0){
                $image_array['default'] = true;
            }
            $lot->images()
                    ->create($image_array);
        }

        return $this->show($lot);
    }
    

    public function destroy(Lot $lot, Lot\Image $image)
    {
        $image->delete();
        return $lot->images()->get();
    }

    /**
     * Make this the default image (thumbnail image)
     *
     * @param Lot $item
     * @param Lot\Image $image
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function setDefault(Lot $lot, Lot\Image $image)
    {
        $lot->images()->update(['default' => 0]);
        $image->update([
            'default' => $image->default ? 0 : 1,
        ]);

        return $this->show($lot);

    }

    /**
     * Allow zooming in on image
     *
     * @param Lot $item
     * @param Lot\Image $image
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function setZoom(Lot $lot, Lot\Image $image)
    {
        $image->update([
            'zoom' => $image->zoom ? 0 : 1,
        ]);

        return $this->show($lot);
    }

    /**
     * Set image as hover image on item small view
     *
     * @param Lot $item
     * @param Lot\Image $image
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function setHover(Lot $lot, Lot\Image $image)
    {
        $lot->images()->update(['hover' => 0]);
        $image->update([
            'hover' => $image->hover ? 0 : 1,
        ]);

        return $this->show($lot);
    }
    
}