<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Auction\PickupMethod;

class LotPickupController extends Controller
{
    /**
     * View Lot Image list
     *
     * @param Request $request
     *
     * @return string
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function index(Request $request)
    {
        $this->page_meta['title'] = 'Methods for delivering lots to ' . config('app.name');

        return template('Auction/admin/lot_pickup/index.twig', [
            'page_meta'      => $this->page_meta,
            'pickup_methods' => PickupMethod::query()
                                            ->latest()
                                            ->get(),
        ]);
    }

    /**
     * View Auction Create form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Lot Pickup Method';

        return template('Auction/admin/lot_pickup/edit.twig', [
            'pickup_method' => new PickupMethod,
            'page_meta'     => $this->page_meta,
            'reminder_types' => PickupMethod::$reminders,
        ]);
    }

    /**
     * Process Auction create request
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name'        => 'required',
            'description' => 'present',
            'reminder'    => 'present',
        ]);

        $method = new PickupMethod();
        $method->fill($validated);
        $method->has_date_picker       = $request->has('has_date_picker');
        $method->has_collection_points = $request->has('has_collection_points');
        $method->save();

        session()->flash('success', 'Method Created');

        return redirect()->to("/admin/lot-pickup/{$method->id}/edit");
    }

    /**
     * View Auction Edit form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function edit($method_id)
    {
        $this->page_meta['title'] = 'Edit Lot Register Delivery Method';
        return template('Auction/admin/lot_pickup/edit.twig', [
            'pickup_method' => PickupMethod::query()->findOrFail($method_id),
            'page_meta' => $this->page_meta,
            'reminder_types' => PickupMethod::$reminders,
        ]);
    }

    /**
     * Process Auction update request
     *
     * @param Request $request
     * @param $method_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $method_id)
    {
        $method = PickupMethod::query()->find($method_id);

        $validated = $this->validate($request, [
            'name' => 'required',
            'description' => 'present',
            'reminder' => 'present',
        ]);

        $method->fill($validated);
        $method->has_date_picker = $request->has('has_date_picker');
        $method->has_collection_points = $request->has('has_collection_points');
        $method->save();


        session()->flash('success', 'Method Updated');
        return redirect()->to("/admin/lot-pickup/{$method->id}/edit");
    }

    /**
     * Delete an auction
     *
     * @param $method_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($method_id)
    {
        $auction = PickupMethod::query()->find($method_id);
        if ($auction) {
            $auction->delete();

            session()->flash('success', 'Method Removed');
        }
        return redirect()->back();
    }
    
}