<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Event;
use Mtc\Auction\Lot\Storage;
use App\Http\Controllers\Controller;
use Mtc\Auction\Lot\StorageLocation;

class LotStorageController extends Controller
{
    /**
     * View Lot Storage list.
     *
     * @param Request $request
     * @return string
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function index(Request $request)
    {
        
        $storage_lots = Storage::with([
            'lot:id,name,reference',
            'auction:id,name',
            'member:id,name',
        ])
                               ->orderBy('storage_expires_at')
                               ->orderBy('created_at')
                               ->get()
                               ->each(function ($storage_lot) {
                                   // The left part is what we assign the value to and the right part comes from the model's accessor
                                   /*if (!is_null($storage_lot->member->name)) {
                                       $storage_lot->member->name = $storage_lot->member->name;
                                   }*/
                               })
                               ->groupBy(function ($lot) {
                                   return $lot->location_id;
                               });
        
        $this->page_meta['title'] = 'Lot Storage';
        
        return template('Auction/admin/storage/index.twig', [
            'page_meta' => $this->page_meta,
            'storage_sections' => $storage_lots,
            'storage_locations' => StorageLocation::get()->pluck('name','id')->prepend('Delete These Storage Entries', 'trash')->prepend('Please Pick a Location to store these lots', '')->toArray(),
        ]);
    }

    /**
     * View Lot Storage list.
     *
     * @param Request $request
     * @return string
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function removed(Request $request)
    {
        $storage_lots = Storage::with([
            'lot:id,name,reference',
            'auction:id,name',
            'member:id, name',
        ])
                               ->onlyTrashed()
                               ->orderBy('deleted_at', 'desc')
                               ->get()
                               ->each(function ($storage_lot) {
                                   // The left part is what we assign the value to and the right part comes from the model's accessor
                                   /*if (!is_null($storage_lot->member->name)) {
                                       $storage_lot->member->name = $storage_lot->member->name;
                                   }*/
                               })
                               ->groupBy(function ($lot) {
                                   return $lot->location_id;
                               });

        $this->page_meta['title'] = 'Lot Storage';
        return template('Auction/admin/storage/removed.twig', [
            'page_meta' => $this->page_meta,
            'storage_sections' => $storage_lots,
            //'storage_locations' => Storage::getLocations(),
        ]);
    }

    /**
     * Edit a storage lot.
     *
     * @param Request $request
     */
    public function edit($storage_lot_id)
    {
        $this->page_meta['title'] = 'Edit Lot Storage Info';

        $storage_lot = Storage::findOrFail($storage_lot_id);
        return template('Auction/admin/storage/edit.twig', [
            'storage_lot' => $storage_lot,
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Update a storage lot.
     *
     * @param Request $request
     */
    public function update(Request $request, $storage_lot_id)
    {
        $storage_lot = Storage::find($storage_lot_id);

        $values = collect($request->all())
            ->only($storage_lot->getFillable())
            ->map(function ($value, $parameter) {
                switch ($parameter) {
                    case 'storage_expires_at':
                        if ((new Carbon)->hasFormat($value, 'd/m/Y')) {
                            $value = Carbon::createFromFormat('d/m/Y', $value)->format('Y-m-d');
                        }

                        break;
                }

                return $value;
            })
            ->all();

        $storage_lot->fill($values);
        $storage_lot->save();

        if ($request->wantsJson()) {
            return [
                'success' => true,
                'message' => 'Storage Lot Updated',
            ];
        }
        session()->flash('success', 'Storage Lot Updated');
        return redirect()->back();
    }

    /**
     * Move lots between sections
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function move(Request $request)
    {
        if ($request->input('location') === 'trash') {
            Storage::query()
                   ->whereIn('id', $request->input('move_lots'))
                   ->delete();

            session()->flash('success', 'Lots Moved to Trash');
            return redirect()->back();
        }

        Storage::query()
               ->withTrashed()
               ->whereIn('id', $request->input('move_lots'))
               ->update([
                            'location_id' => $request->input('location') ?: null,
                            'deleted_at' => null,
                        ]);

        session()->flash('success', 'Lots Moved');
        return redirect()->back();
    }
}