<?php

namespace Mtc\Auction\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Auction\Auction;
use Mtc\Auction\AuctionMember;
use Mtc\Auction\Http\Response\Lotlist;
use Mtc\Auction\Http\Response\MyBidList;
use Mtc\Auction\Lot;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Models\Country;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Members\HasMembersNavigation;
use Mtc\Members\Member;
use Mtc\Orders\Order;

class AuctionMemberController extends Controller
{
    use HasMembersNavigation;
    const PER_PAGE = 80;

    /**
     * Sellers Lots
     *
     * @param Request $request
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function sellerLots(Request $request)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;
        
        if (!$this->member->exists) {
            return redirect()->to('/members/login');
        }

        $auction_member = AuctionMember::mutateMemberObject($this->member);

        if (!$auction_member->auction_fields->seller_membership_fee_paid
            && (config('auction.settings.registration_fee_for_sellers'))) {
            session()->flash('error', __('auction.account.must_pay_registration_fee'));
            return redirect()->back();
        }

        if ($request->wantsJson()) {
            $collection = Lot::query()
                             ->with([
                                 'latestBid',
                                 'defaultImage',
                                 'hoverImage',
                             ])
                             ->setAuctionId($request->input('filters.auction_id'))
                             ->setStatusFilter($request->input('filters.status'))
                             ->setListingSort($request->input('filters.order'))
                             ->where('seller_id', $this->member->id)
                             ->get();

            return (new Lotlist($collection))->handle(true, true);
        }

        $show_relist_return_button = Lot::query()
                                        ->where('seller_id', $this->member->id)
                                        ->where('status', Lot\Status::NOT_SOLD)
                                        ->exists();

        $auctions = Auction::query()
                           ->latest('ends_at')
                           ->get();

        $default_filters = new \stdClass();

        if (empty($default_filters->auction_id = $auctions->where('is_live', 1)->first())) {
            $default_filters->auction_id = $auctions->where('ends_at', '<', Carbon::now())->first()->id;
        }

        $this->page_meta['page_title'] = "My Lots";
        $this->page_meta['title'] = "My Lots | " . config('app.name');

        app('twig')->addGlobal("is_members_area", true);
        
        return template('Auction/members/my-lots.twig', [
            'page_meta' => $this->page_meta,
            'per_page' => self::PER_PAGE,
            'sort_options' => Lot::getListSortOptionsFlat(),
            'auctions' => $auctions->pluck('name', 'id'),
            'show_relist_return_button' => $show_relist_return_button,
            'default_filters' => $default_filters,
            'member' => $this->member,
            'sections' => $this->sections(),
        ]);
    }

    /**
     * Users Bids
     *
     * @param Request $request
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function myBids(Request $request)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;
        
        if (!$this->member->exists) {
            return redirect()->to('/members/login');
        }

        /*if (!$this->member->can_bid) {
            session()->flash('error', __('auction.account.must_pay_registration_fee'));
            return redirect()->back();
        }*/

        if ($request->wantsJson()) {
            $collection = Lot::query()
                             ->withHighestBidder()
                             ->withBidValue(false)
                             ->setAuctionId($request->input('filters.auction_id'))
                             ->whereHas('bids', function ($bid_query) {
                                 return $bid_query->where('member_id', $this->member->id);
                             });

            switch ($request->input('filters.status')) {
                case 'winning':
                    $collection->having('highest_bidder', $this->member->id)
                               ->havingRaw('`bid_value` >= `auction_lots`.`reserve_price`');
                    break;
                case 'losing':
                    $collection->having('highest_bidder', '!=', $this->member->id)
                               ->orHavingRaw('`bid_value` < `auction_lots`.`reserve_price`');
                    break;
            }

            $collection = $collection->orderBy('ends_at', 'desc')
                                     ->orderBy('id', 'desc')
                                     ->get();

            return (new MyBidList($collection))->handle();
        }

        $this->page_meta['page_title'] = "My Bids";
        $this->page_meta['title'] = "My Bids | " . config('app.name');

        app('twig')->addGlobal("is_members_area", true);
        
        return template('Auction/members/my-bids.twig', [
            'page_meta' => $this->page_meta,
            'per_page' => self::PER_PAGE,
            'auctions' => Auction::query()->where('starts_at', '<', Carbon::now())->latest('ends_at')->get(),
            'current_live_auction' => Auction::currentLiveAuction(),
            'member' => $this->member,
            'sections' => $this->sections(),
        ]);
    }

/**
* Bids user is winning
*
* @param Request $request
* @throws \Twig_Error_Runtime
* @throws \Twig_Error_Loader
* @throws \Twig_Error_Syntax
*/
    public function myWinningBids(Request $request)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;
        
        if (!$this->member->exists) {
            return redirect()->to('/members/login');
        }

        $collection = Lot::query()
                         ->withHighestBidder()
                         ->withBidValue(false)
                         ->whereHas('bids', function ($bid_query) {
                             return $bid_query->where('member_id', $this->member->id);
                         })
                         ->having('highest_bidder', $this->member->id)
                         ->havingRaw('`bid_value` >= `auction_lots`.`reserve_price`')
                         ->where('status', Lot\Status::LIVE)
                         ->orderBy('ends_at', 'desc')
                         ->orderBy('id', 'desc')
                         ->get();

        return (new LotList($collection))->handle();
    }

    /**
     * Items user has won
     *
     * @param Request $request
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function myWinnings(Request $request)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;
        
        if (!$this->member->exists) {
            return redirect()->to('/members/login');
        }
        
        /*if (!$this->member->can_bid) {
            session()->flash('error', __('auction.account.must_pay_registration_fee'));
            return redirect()->back();
        }*/

        if ($request->wantsJson()) {
            $collection = Lot::query()
                             ->latest()
                             ->where('buyer_id', $this->member->id)
                             ->get();

            return (new LotList($collection))->handle();
        }
        

        $this->page_meta['page_title'] = "My Winnings";
        $this->page_meta['title'] = "My Winnings | " . config('app.name');

        app('twig')->addGlobal("is_members_area", true);
        
        return template('Auction/members/my-winnings.twig', [
            'page_meta' => $this->page_meta,
            'per_page' => self::PER_PAGE,
            'member' => $this->member,
            'sections' => $this->sections(),
        ]);
    }

    public function receipt($lot_id)
    {

        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;
        if (!$this->member->exists && empty($_SESSION['adminId'])) {
            return redirect()->to('/members/login?redirect=/members/my-lots/' . $lot_id . '/receipt');
        }

        // Site might have custom numbering for lots
        $lot_id = Lot::decodeLotIdToId($lot_id);

        $lot = Lot::query()->findOrFail($lot_id);
        if ($lot->seller_id != $this->member->id && empty($_SESSION['adminId'])) {
            return redirect()->back();
        }

        $lots = Lot::query()
            ->where('seller_id', $lot->seller_id)
            ->where('auction_id', $lot->auction_id)
            ->whereIn('status', Lot\Status::$winning_statuses)
            ->get();

        $fees = collect();

        $seller = Member::query()->find($lot->seller_id);
        $seller_address = \Mtc\Auction\Member\Member::addressBilling($lot->seller_id);
        $no_vat = !Country::isEu($seller_address->country_code);

        // Only allow JE & GY postcodes for UK based no_vat zones
        $postcode = strtoupper(substr($seller_address->postcode, 0, 2));
        if ($seller_address->country_code === 'GB' && in_array($postcode, array('JE', 'GY'))) {
            $no_vat = true;
        }

        $sum_hammer_price = $lots->sum('highest_bid');
        $sum_commission = $no_vat ? $lots->sum('seller_commission') : $lots->sum('seller_commission_with_vat');
        $sum_fees = 0;
        foreach ($lots as $lot) {
            if (empty(floatval($lot->reserve_price))) {
                $lot->fees = $lot->fees()->where(function ($query) {
                    $query->where('PLU', '!=', config('auction.lots.purchase_plu_defaults.LOT_RESERVE_PLU'))
                        ->orWhere('PLU', null);
                })->get();
                $lot_fees = $lot->fees()->where(function ($query) {
                    $query->where('PLU', '!=', config('auction.lots.purchase_plu_defaults.LOT_RESERVE_PLU'))
                        ->orWhere('PLU', null);
                })->get();
            } else {
                $lot->fees = $lot->fees()->get();
                $lot_fees = $lot->fees()->get();
            }
            $sum_fees += $lot_fees->sum('amount');
            $fees = $fees->merge($lot_fees);
        }
        $sum_proceeds = $sum_hammer_price - $sum_commission - $sum_fees;

        $fees = $fees->groupBy('description');

        $template_name = ENVIRONMENT === 'production' ? 'Auction/members/receipt_old.twig' : 'Auction/members/receipt.twig';

        return template($template_name, [
            'lots' => $lots,
            'fees' => $fees,
            'sum_hammer_price' => $sum_hammer_price,
            'sum_commission' => $sum_commission,
            'sum_fees' => $sum_fees,
            'sum_proceeds' => $sum_proceeds,
//            'basket_countries' => App::make('basket_countries'),
            'no_vat' => $no_vat,
            'seller' => $seller,
            'seller_address' => $seller_address,
            'payout_ref' => $lot->payout_referene,
            'payout_date' => $lot->payout_made_at,
            'member' => Member::find($lot->seller_id),
        ]);
    }

    /**
     * Render invoice for lot
     *
     * @param $lot_id
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function invoice(Request $request, $lot_id)
    {

        $order_id = $request->input('order_id');

        if (empty($order_id)) {
            // Site might have custom numbering for lots
            $lot_id = \Mtc\Auction\Lot::decodeLotIdToId($lot_id);
            $lot = Lot::query()->findOrFail($lot_id);
            $order_id = $lot->buy_order_id;
            
        }

        $lots_in_invoice = Lot::query()
                              ->where('buy_order_id', $order_id)
                              ->get();
        $order = Order::find($lot->buy_order_id);
        $items = $order->items()->get();
        
        $delivery_cost = $order->deliverySurcharge;

        $sum_vat_on_buyers_commission = $lots_in_invoice->sum('buyer_commission_with_vat');
        $sum_vat_on_hammer_price = $lots_in_invoice->sum('highest_bid');
        $vat_total = $order->vat_value;

        $this->page_meta['page_title'] = "Invoice";
        $this->page_meta['title'] = "Invoice | " . SITE_NAME;
        return template('Auction/members/buyer_invoice.twig', [
            'order' => $order,
            'order_items' => $items,
            'lots' => $lots_in_invoice,
            'delivery_cost' => $delivery_cost,
            //            'basket_countries' => app('basket_countries'),
            'subtotal' => $order->cost_subtotal,
            'vat_total' => $vat_total,
            'member_view' => empty($_SESSION['adminId']),
        ]);
    }

}