<?php

namespace Mtc\Auction\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Event;
use Mtc\Auction\AuctionMember;
use Mtc\Auction\Bid;
use Mtc\Auction\Events\BidUpdate;
use Mtc\Auction\Lot;
use Mtc\Auction\Lot\Status;
use Mtc\Auction\Watchlist;
use Mtc\Members\Facades\MemberAuth;

class BidController extends Controller
{
    /**
     * View Lot
     *
     * @param $lot_id
     */
    public function show($lot_id)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;

        //Mutate Logged in Member to the AuctionMember Class
        if (!is_null($this->member)) {
            $this->member = AuctionMember::mutateMemberObject($this->member);
        }


        /** @var Lot $lot */
        $lot = Lot::query()
                  ->with([
                             'latestBid',
                         ])
                  ->find($lot_id);

        $max_bid = false;

        $own_lot = false;

        $is_watching = false;

        $user_has_bids = false;

        if (!is_null($this->member)) {
            $this->member = AuctionMember::find($this->member->id)
                                         ->first();
            ($this->member->id);
            if (!is_null($lot->myLatestBid($this->member->id))) {
                $max_bid = $lot->myLatestBid($this->member->id)->max_bid;
            }

            if ($lot->seller_id === $this->member->id) {
                $own_lot = true;
            }

            $is_watching = Watchlist::isUserWatching($lot_id, $this->member);

            $user_has_bids = $this->member->userHasBidsOnLot($lot_id);
        }

        $owner = null;

        if (!is_null($lot->latestBid)) {
            $owner = $lot->latestBid->member->hashed_id;
        }

        return [
            'bid_value'                       => $lot->latestBid->bid ?? $lot->starting_price,
            'buy_out_price'                   => $lot->buy_out_price,
            'max_bid'                         => $max_bid,
            'owner'                           => $owner,
            'ends_at'                         => $lot->ends_at,
            'has_reserve'                     => !empty(floatval($lot->reserve_price)),
            'reserve_met'                     => $lot->reserve_met,
            'can_bid'                         => (!is_null($this->member)) ? $lot->canUserBidOnThisLot(
                $this->member
            ) : false,
            'user_must_pay_buyers_membership' => (config('auction.settings.registration_fee_for_bidders')
                                                  && (!is_null(
                    $this->member
                ))
                                                  && (is_null($this->member->auction_fields) ||  $this->member->auction_fields->buyer_membership_fee_paid)) ? true : false,
            'own_lot'                         => $own_lot,
            'not_allowed_to_bid'              => $lot->not_allowed_to_bid,
            'time_remaining'                  => $lot->time_remaining,
            'is_watching'                     => $is_watching,
            'is_biddable'                     => $lot->isBiddable(),
            'is_before_live'                  => in_array($lot->status, Status::$pre_auction_statuses),
            'is_after_live'                   => in_array($lot->status, Status::$past_statuses),
            'is_live'                         => $lot->status == Status::LIVE,
            'is_logged_in'                    => MemberAuth::user(),
            'user_has_bids'                   => $user_has_bids,
            'next_bid_increment'              => $lot->next_bid_increment,
            'bid_descriptor_value'            => $lot->bid_descriptor_value,
            'bid_descriptor_text'             => $lot->bid_descriptor_text,
            'total_bids'     => $lot->getTotalBids(),
            'total_watchers' => $lot->getTotalWatchers(),
            'step_size' => Bid\Rule::getStep($lot->latestBid->bid ?? $lot->starting_price ?? 0.00),
            'step_sizes' => Bid\Rule::query()->get()->jsonSerialize(),
            'reserve_price' => $lot->reserve_price,
            'initial_bid' => $lot->getInitialBidValue()
        ];
    }

    /**
     * View Lot
     *
     * @param $lot_id
     */
    public function store(Request $request, $lot_id)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;

        //Mutate Logged in Member to the AuctionMember Class
        if (!is_null($this->member)) {
            $this->member = AuctionMember::mutateMemberObject($this->member);
        }

        try {
            $validated = $this->validate($request, [
                'bid' => 'numeric',
            ]);

            $hide_outbid_question = true;
            /*if ($request->input('outbid_notification') === 'true') {
                $settings = $this->member->notificationSettings ?? new MemberNotification();
                $settings->member_id = $this->member->id;
                $settings->outbid_notification = 1;
                $settings->save();
                $hide_outbid_question = 1;
            }*/

            $validated['bid_in_base_currency'] = Bid::convertToBaseCurrency(
                $validated['bid'],
                $_SESSION['selected_currency']
            );

            $bid = new Bid();
            $bid->fill([
                           'lot_id'    => $lot_id,
                           'member_id' => $this->member->id,
                           'max_bid'   => $validated['bid_in_base_currency'],
                       ]);

            $resulting_bid = $bid->tryPlacingBid();

            Event::dispatch(new BidUpdate($resulting_bid));

            /** @var Lot $lot */
            $lot = Lot::query()
                      ->with([
                                 'latestBid',
                             ])
                      ->find($lot_id);

            return [
                'bid_value'                       => $lot->latestBid->bid ?? $lot->starting_price ?? 0,
                'buy_it_now_price'                => $lot->buy_it_now_price ?? 0,
                'max_bid'                         => (!is_null(
                    $lot->myLatestBid($this->member->id)
                )) ? $lot->myLatestBid(
                    $this->member->id
                )->max_bid ?: false : false,
                'owner'                           => $this->member->hashed_id,
                'ends_at'                         => $lot->ends_at,
                'has_reserve'                     => !empty(floatval($lot->reserve_price)),
                'reserve_met'                     => $lot->reserve_met,
                'time_remaining'                  => $lot->time_remaining,
                'can_bid'                         => (!is_null($this->member)) ? $lot->canUserBidOnThisLot(
                    $this->member
                ) : false,
                'user_must_pay_buyers_membership' => (config('auction.settings.registration_fee_for_bidders')
                                                      && (!is_null($this->member))
                                                      && (is_null($this->member->auction_fields) || $this->member->auction_fields->buyer_membership_fee_paid)) ? true : false,
                'own_lot'                         => $lot->seller_id === $this->member->id,
                'is_biddable'                     => $lot->isBiddable(),
                'is_before_live'                  => in_array($lot->status, Status::$pre_auction_statuses),
                'is_after_live'                   => in_array($lot->status, Status::$past_statuses),
                'is_live'                         => $lot->status == Status::LIVE,
                'is_logged_in'                    => MemberAuth::check(),
                'user_has_bids'                   => $this->member->userHasBidsOnLot($lot_id),
                'next_bid_increment'              => $lot->next_bid_increment,
                'bid_descriptor_value'            => $lot->bid_descriptor_value,
                'bid_descriptor_text'             => $lot->bid_descriptor_text,
                'hide_outbid_question'            => $hide_outbid_question,
                'reserve_price' => $lot->reserve_price ?? 0,
                'total_bids'     => $lot->getTotalBids(),
                'total_watchers' => $lot->getTotalWatchers(),
                'message'                         => [
                    'type'    => 'success',
                    'content' => __('auction::lots.bidding.bid_placed', [
                        'max_bid'   => Bid::setInUsersCurrency($resulting_bid->max_bid),
                        'bid_value' => Bid::setInUsersCurrency($resulting_bid->bid),
                    ]),
                ],
            ];
        } catch (\Exception $exception) {
            /** @var Lot $lot */
            $lot = Lot::query()
                      ->with([
                                 'latestBid',
                             ])
                      ->find($lot_id);

            Event::dispatch(new BidUpdate($lot->latestBid ?? new Bid()));

            return [
                'bid_value'                       => $lot->latestBid->bid ?? $lot->starting_price,
                'buy_it_now_price'                => $lot->buy_it_now_price,
                'max_bid'                         => $lot->myLatestBid($this->member->id) ? $lot->myLatestBid(
                    $this->member->id
                )->max_bid : false,
                'owner'                           => $lot->latestBid->member->hashed_id ?? null,
                'ends_at'                         => $lot->ends_at,
                'has_reserve'                     => !empty(floatval($lot->reserve_price)),
                'reserve_met'                     => $lot->reserve_met,
                'time_remaining'                  => $lot->time_remaining,
                'can_bid'                         => (!is_null($this->member)) ? $lot->canUserBidOnThisLot(
                    $this->member
                ) : false,
                'user_must_pay_buyers_membership' => (config('auction.settings.registration_fee_for_bidders')
                                                      && (!is_null($this->member))
                                                      && (is_null($this->member->auction_fields) || $this->member->auction_fields->buyer_membership_fee_paid)) ? true : false,
                'own_lot'                         => $lot->seller_id === $this->member->id,
                'is_biddable'                     => $lot->isBiddable(),
                'is_before_live'                  => in_array($lot->status, Status::$pre_auction_statuses),
                'is_after_live'                   => in_array($lot->status, Status::$past_statuses),
                'is_live'                         => $lot->status == Status::LIVE,
                'is_logged_in'                    => MemberAuth::check(),
                'user_has_bids'                   => $this->member->userHasBidsOnLot($lot_id),
                'next_bid_increment'              => $lot->next_bid_increment,
                'bid_descriptor_value'            => $lot->bid_descriptor_value,
                'bid_descriptor_text'             => $lot->bid_descriptor_text,
                'reserve_price' => $lot->reserve_price ?? 0,
                'total_bids'     => $lot->getTotalBids(),
                'total_watchers' => $lot->getTotalWatchers(),
                'message'                         => [
                    'type'    => 'error',
                    'content' => $exception->getMessage(),
                ],
            ];
        }
    }
}