<?php

namespace Mtc\Auction\Http\Controllers;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Mtc\Auction\AuctionMember;
use Mtc\Auction\Http\Response\Lotlist;
use Mtc\Auction\Lot;
use Mtc\Auction\Auction;
use Mtc\Auction\Filter;
use Illuminate\Support\Facades\App;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Core\Currency;
use Mtc\Members\Facades\MemberAuth;

class LotController extends Controller
{

    /**
     * View Lot
     *
     * @param $lot_id
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function show(Request $request, $lot_id, $lot_name = '', $search = '', $previous_search = '')
    {
        if (!empty($previous_search)) {
            $previous_search_array            = explode('&', $previous_search);
            $previous_search_params_formatted = [];
            foreach ($previous_search_array as $value) {
                switch (substr($value, 0, strpos($value, '-'))) {
                    case 'c':
                        $previous_search_params_formatted['active_filters']['country'] = [
                            urlencode(substr($value, (strpos($value, '-') + 1))),
                        ];
                        break;
                    case 'a':
                        $previous_search_params_formatted['active_filters']['age'] = [
                            urlencode(substr($value, (strpos($value, '-') + 1))),
                        ];
                        break;
                    case 'r':
                        $previous_search_params_formatted['active_filters']['region'] = [
                            urlencode(substr($value, (strpos($value, '-') + 1))),
                        ];
                        break;
                    case 'lt':
                        $previous_search_params_formatted['active_filters']['lot_type'] = [
                            urlencode(substr($value, (strpos($value, '-') + 1))),
                        ];
                        break;
                    case 'd':
                        $previous_search_params_formatted['active_filters']['distillery'] = [
                            urlencode(substr($value, (strpos($value, '-') + 1))),
                        ];
                        break;
                    case 'q':
                        $previous_search_params_formatted['search_filter'] = urlencode(substr($value,
                            (strpos($value, '-') + 1)));
                        break;
                    case 'sb':
                        $previous_search_params_formatted['sort_by'] = urlencode(substr($value,
                            (strpos($value, '-') + 1)));
                        break;
                    case 'au':
                        $previous_search_params_formatted['auction'] = urlencode(substr($value,
                            (strpos($value, '-') + 1)));
                        break;
                }
            }
        }
        
        // Site might have custom numbering for lots
        $lot_id = Lot::decodeLotIdToId($lot_id);

        /** @var Lot $lot */
        $lot = Lot::query()
                  ->with([
                      'custom',
                      'defaultImage',
                      'images',
                  ])
                  ->find($lot_id);

        

        $lot->incrementViewCounter();

        Lot\Recent::add($lot->id);
        $lot->recently_viewed = Lot\Recent::get($lot->id);
        $lot->complementary   = $lot->getComplementary();

        $member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;

        //Mutate Logged in Member to the AuctionMember Class
        if(!is_null($member)){
            $member = AuctionMember::mutateMemberObject($member);
        }
        
        $lot->can_bid            = (!is_null($member)) ? $lot->canUserBidOnThisLot($member) : false;
        
        //Confirm the user has paid any membership fees
        $lot->user_must_pay_buyers_membership = (config('auction.settings.registration_fee_for_bidders')
                  && (!is_null(
                    $member
                )) && (is_null($member->auction_fields) || !$member->auction_fields->buyer_membership_fee_paid));

        if (!empty($previous_search_params_formatted)) {
            $next_previous_buttons = Filter::filterWithArrayAndLot($lot->lot_id, $previous_search_params_formatted);

            if (!empty($next_previous_buttons['next'])) {
                $next_button = $next_previous_buttons['next']->url . '/searched/' . $previous_search;
            }
            if (!empty($next_previous_buttons['previous'])) {
                $previous_button = $next_previous_buttons['previous']->url . '/searched/' . $previous_search;
            }
        } else {
            if (!empty($lot->nextLotInAuction())) {
                $next_button = $lot->nextLotInAuction()->url . '/' . $search;
            }

            if (!empty($lot->previousLotInAuction())) {
                $previous_button = $lot->previousLotInAuction()->url . '/' . $search;
            }
        }

        $other_currencies = [];
        $currencies       = Currency::where('ratio', '>', '0')
                                    ->pluck('currency')
                                    ->toArray();
        if (count($currencies) > 0) {
            $other_currencies = $currencies;
        }

        $breadcrumbs = [
            [
                'id' => 1,
                'url' => '/',
                'name' => 'Home',
            ]
        ];

        if (!is_null($lot->auction) && $lot->auction->is_live) {
            $breadcrumbs[] = [
                'id'   => 'auction_' . $lot->auction->id,
                'url'  => '/live-auction',
                'name' => $lot->auction->name,
            ];
        } elseif (!is_null($lot->auction)) {
            $breadcrumbs[] = [
                'id'   => 1,
                'url'  => '/past-auctions',
                'name' => 'Past Auctions',
            ];
            $breadcrumbs[] = [
                'id'   => 'auction_' . $lot->auction->id,
                'url'  => $lot->auction->url,
                'name' => $lot->auction->name,
            ];
        }
        $breadcrumbs[] = [
            'id' => 'lot_' . $lot->id,
            'url' => '',
            'name' => $lot->name,
        ];

        $lot->initial_bid = $lot->getInitialBidValue();
        
        $this->page_meta['title'] = $lot->name . ' | ' . env('APP_NAME');
        
        $default_image = $lot->defaultImage;
        
        if(is_null($default_image)){
            $default_image = $lot->images->first();
        }

        return template('Auction/lot/full.twig', [
            'page_meta'                    => $this->page_meta,
            'lot'                          => $lot,
            'default_image'                          => $default_image,
            //'default_lot_image_path' => Lot\Image::DEFAULT_IMAGE_PATH,
            //'display_custom_fields' => $custom_fields,
            'display_additional_fields'    => Lot::getAdditionalDisplayFields(),
            'lot_past_statuses'            => Lot\Status::$past_statuses,
            'live_status'                  => Lot\Status::LIVE,
            'not_sold_status'              => Lot\Status::NOT_SOLD,
            'member_notification_settings' => $member->notificationSettings ?? [],
            'member_id' => $member->id ?? 0,
            //'distillery' => $distillery,
            'search'                       => $search,
            /*
            'formatted_filter' => $previous_search_params_formatted ?? [],
            'raw_filter' => $previous_search,
            */ 'next_button'   => $next_button ?? '',
            'previous_button'              => $previous_button ?? '',
            'complementary_url'            => $lot->url . '/complementary',
            'currencies'                   => Currency::getEnabledCurrencies(),
            //'last_auction_url' => $_SESSION['last_auction_url'],
            'auction_status'               => (!is_null($lot->auction)) ? $lot->auction->getAdminTypeAttribute() : null,
            //'translation' => $translation_array,
            'other_currencies'             => $other_currencies,
            'breadcrumbs' => $breadcrumbs
        ]);
    }

    /**
     * View Lot Complementary Items
     *
     * @param int $lot_id
     */
    public function getComplementary($lot_id)
    {
        // Site might have custom numbering for lots
        $lot_id = Lot::decodeLotIdToId($lot_id);

        /** @var Lot $lot */
        $lot = Lot::find($lot_id);

        return (new Lotlist($lot->getComplementary()))->handle();
    }


    /**
     * Add lots user has won to basket and redirect to the basket
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function buyLots(Request $request)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;

        if (!$this->member->exists) {
            return redirect()->to('/members/login?redirect=/purchase-lots');
        }

        self::addWonLotsToBasket();

        return redirect()->to('/basket');
    }

    public static function addWonLotsToBasket()
    {
        $member           = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;
        $basket_interface = App::make(BasketRepositoryInterface::class);
        //        $basket_id = session();

        Lot::query()
           ->where('buyer_id', $member->id)
           ->where('status', Lot\Status::SOLD)
           ->whereNull('buy_order_id')
           ->get()
           ->each(function (Lot $lot) use ($basket_interface) {
               if (!$basket_interface->hasItem($lot)) {
                   $basket_interface->addItem($lot, 1, true, false);
               }
               //The late fees object is best to be created for sanity's sake as the surcharge object needs to exist
               $lot->lateFees()->firstOrCreate(['lot_id'=>$lot->id]);
               
               if($lot->ends_at->diff(Carbon::now())->days >= config('auction.late_payment_fee.payment_workdays') && config('auction.settings.late_lot_payment_fees_enabled')){
                   $basket_interface->addSurcharge($lot->lateFees->id, Lot\LotLatePaymentFee::class);
               } elseif($basket_interface->hasSurcharge($lot->lateFees->id, Lot\LotLatePaymentFee::class)) {
                   $basket_interface->removeSurchargeByIDAndType($lot->lateFees->id, Lot\LotLatePaymentFee::class);
               }
           });
    }
}