<?php

namespace Mtc\Auction\Http\Response;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\Auction\AuctionMember;
use Mtc\Auction\Bid\Rule;
use Mtc\Auction\Lot;
use Mtc\Auction\Watchlist;
use Illuminate\Support\Facades\Auth;
use Mtc\Members\Facades\MemberAuth;

class Lotlist
{

    /**
     * list of lots
     *
     * @var Collection $lot_list
     */
    protected $lot_list;

    /**
     * LotList constructor.
     *
     * @param Collection $lot_list
     */
    public function __construct(Collection $lot_list)
    {
        $this->lot_list = $lot_list;
    }

    /**
     * Update collection to have all necessary data
     *
     * @param bool $owner_info whether to show sensitive information
     * @return Collection
     */
    public function handle($owner_info = false, $force_default_thumb_pre_auction = false)
    {
        $member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;

        //Mutate Logged in Member to the AuctionMember Class
        if(!is_null($member)){
            $member = AuctionMember::mutateMemberObject($member);
        }
        
        return $this->lot_list
            ->each(function (Lot $lot) use ($owner_info, $force_default_thumb_pre_auction, $member) {
                $lot->setHidden($lot->hidden_from_lot_lists);

                $locale = App::getLocale();

                $lot->has_reserve = !empty(floatval($lot->reserve_price));

                $lot->reserve_met = true;
                if($lot->has_reserve){
                    if(!is_null($lot->latestBid)){
                        $lot->reserve_met = $lot->reserve_price <= ($lot->latestBid->bid ?? 0);
                    } else {
                        $lot->reserve_met = false; 
                    }
                }

                // TODO: Placeholder Image
                $lot->current_bid = $lot->latestBid->bid ?? $lot->starting_price;
                $lot->max_bid = (!is_null($member) && !is_null($lot->myLatestBid($member->id))) ? $lot->myLatestBid($member->id)->max_bid ?: false : false;
                $lot->owner = (!is_null($lot->latestBid)) ? $lot->latestBid->member->hashed_id : null;
                $lot->thumb_image = $lot->defaultImage ? $lot->defaultImage->getUrl('thumbs') : config('auction.default_lot_image_path');
                $lot->hover_thumb_image = $lot->hoverImage ? $lot->hoverImage->getUrl('thumbs') : '';
                $lot->large_image = $lot->defaultImage ? $lot->defaultImage->getUrl('large') : config('auction.default_lot_image_path');
                $lot->distillery_year = date('Y', strtotime($lot->custom->distillery_date));
                $lot->url = $lot->getUrlAttribute();
                $lot->lot_id = $lot->getLotIdAttribute();
                $lot->age = $lot->custom->age;
                $lot->time_left = Carbon::now()->diffInSeconds($lot->ends_at, false);
                $lot->status_text = Lot\Status::$lot_statuses[$lot->status];
                $lot->can_bid = !is_null($member) && $lot->canUserBidOnThisLot($member);
                $lot->own_lot = !is_null($member) && $lot->seller_id === $member->id;
                $lot->is_watching = !is_null($member) && Watchlist::isUserWatching($lot->id, $member);
                $lot->is_sold = in_array($lot->status, Lot\Status::$winning_statuses) ? 1 : 0;
                $lot->is_biddable = $lot->isBiddable();
                $lot->is_before_live = in_array($lot->status, Lot\Status::$pre_auction_statuses);
                $lot->is_after_live = in_array($lot->status, Lot\Status::$past_statuses);
                if ($force_default_thumb_pre_auction && $lot->is_before_live) {
                    $lot->thumb_image = $lot->hover_thumb_image = config('auction.default_lot_image_path');
                }
                $lot->is_live = $lot->status == Lot\Status::LIVE;
                $lot->is_logged_in = Auth::id()>0;
                $lot->user_has_bids = !is_null($member) && $member->userHasBidsOnLot($lot->id);
                $lot->bid_descriptor_value = $lot->bid_descriptor_value;
                $lot->bid_descriptor_text = $lot->bid_descriptor_text;
                $lot->next_bid_increment = $lot->next_bid_increment;
                $lot->total_bids = $lot->getTotalBids();
                $lot->total_watchers = $lot->getTotalWatchers();

                $lot->step_size = Rule::getStep($lot->latestBid->bid ?? $lot->starting_price ?? 0.00);
                $lot->step_sizes = Rule::query()->get()->jsonSerialize();
                $lot->initial_bid = $lot->getInitialBidValue();
                
                if ($owner_info) {
                    $lot->makeVisible('reserve_price');
                    $lot->next_bid_floor = $lot->latestBid ? Rule::findNextBidStep($lot->latestBid, $lot->latestBid->bid) : 0;
                }
                
                //Translations
                if ($locale !== 'en') {
                    //check if this lot has a translation
                    $translation = $lot->translations()
                                       ->whereHas('language', function ($query) use ($locale) {
                                           $query->where('code', $locale);
                                       })
                                       ->first();
                    if (is_numeric($translation->id)) {
                        //if a translation exists for the locale, use the translations
                        $lot->default_name          = $lot->name;
                        $lot->name          = $translation->name;
                        $lot->description   = $translation->description;
                        $lot->description_2 = $translation->description_2;
                    }
                }
            });
    }
}