<?php

namespace Mtc\Auction;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Mtc\Auction\Bid\Rule;
use Mtc\Auction\Filter\SortByLotId;
use Mtc\Auction\Filter\SortByPrice;
use Mtc\Auction\Lot\Custom;
use Mtc\Auction\Lot\CustomMultiple;
use Mtc\Auction\Lot\Fee;
use Mtc\Auction\Lot\LotInsurance;
use Mtc\Auction\Lot\LotLatePaymentFee;
use Mtc\Auction\Lot\SellerDelivery;
use Mtc\Auction\Lot\SellerPayout;
use Mtc\Auction\Lot\Status;
use Mtc\Auction\Lot\Storage;
use Mtc\Basket\BasketItemPricing;
use Mtc\Basket\Contracts\Purchasable;
use Mtc\Members\Member;
use Mtc\Auction\Lot\HasOrders;
use Mtc\Auction\Lot\EndProcess;
use Mtc\Auction\Lot\Image;
use Mtc\Auction\Lot\HasComplementary;
use Mtc\Modules\Auction\Classes\Filter\SortByName;
use Mtc\Money\Events\LoadPriceModifiers;
use Mtc\Money\HasPrices;
use Mtc\Money\Price;
use Mtc\Money\PriceModifier;
use Mtc\ShippingManager\Events\SetDeliveryForBasket;
use Mtc\Shop\Brand;


class Lot extends Model implements Purchasable
{

    use HasComplementary, EndProcess, HasOrders, HasPrices;

    /**
     * Table name
     *
     * @var string $table
     */
    protected $table = 'auction_lots';

    /**
     * Mass assignable attributes
     *
     * @var array $fillable
     */
    protected $fillable = [
        'id',
        'relisted_from_lot_id',
        'auction_id',
        'seller_id',
        'buyer_id',
        'brand_id',
        'register_order_id',
        'buy_order_id',
        'price_trends_id',
        'published',
        'ends_at',
        'reference',
        'status',
        'name',
        'description',
        'short_description',
        'other_information',
        'condition',
        'is_in_slideshow',
        'is_featured',
        'weight',
        'reserve_price',
        'starting_price',
        'buy_out_price',
        'with_insurance',
        'payout_details',
        'payout_method',
        'payout_made_at',
        'payout_reference',
        'hammer_price',
        'buyer_commission',
        'buyer_commission_percentage',
        'seller_commission',
        'seller_commission_percentage',
        'delivery_method_id',
        'delivery_method_date',
        'collection_point_name',
        'barcode',
        'seller_commission_limit',
        'price_trends',
        'custom_field_set_id',
        'lot_condition',
        'relisting_enabled',
        //'condition_text',
    ];

    public $hidden_from_lot_edit_emails = [
        'seller_commission_limit',
        'ignore_late_payment_fee',
        'num_views',
        'last_view_at',
    ];

    public $hidden_from_lot_lists = [
        'relisted_from_lot_id',
        'barcode',
        'seller_id',
        'buyer_id',
        'registration_fee',
        'register_order_id',
        'buy_order_id',
        'published',
        'description',
        'description_2',
        'condition',
        'weight',
        'reserve_price',
        'with_insurance',
        'is_featured',
        'is_in_slideshow',
        'courier_details',
        'payout_details',
        "payout_made_at",
        'payout_reference',
        'delivery_method_id',
        'delivery_method_date',
        'collection_point_name',
        'hammer_price',
        "hammer_price_with_vat",
        "buyer_commission_percentage",
        "buyer_commission",
        "buyer_commission_with_vat",
        "seller_commission_percentage",
        "seller_commission",
        "seller_commission_with_vat",
        "seller_commission_limit",
        "created_at",
        "updated_at",
        'ignore_late_payment_fee',
        'num_views',
        'last_view_at',
        'price_trends',
        'historic_featured',
        'auction',
        'custom',
        'latestBid',
        'relisting_enabled',
    ];
    
    protected $appends = [
      'url',
      'show_invoice_link',
    ];
    
    /**
     * Cast variables to types
     *
     * @var array $casts
     */
    protected $casts = [
        'ends_at' => 'datetime',
        'original_ends_at' => 'datetime',
        'payout_made_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'last_view_at' => 'datetime',
        ''
    ];

    /**
     * Fields that are cast to prices
     *
     * @var array
     */
    protected $price_fields = [
        'price',
    ];

    //protected $tax_attribute = 'vat_rate';

    /**
     * Relationship with Auction
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function auction()
    {
        return $this->belongsTo(Auction::class);
    }
    
    /**
     * Relationship with Auction
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasOne
     */
    public function price_trend()
    {
        return $this->hasOne(PriceTrend::class);
    }
    
    /**
     * Relationship with Brand
     * 
     * Previous Versions would set Distillery/Game/Manufacturer as a Custom field
     * Since this version of the Auction System could be used with the Shop more
     * Sharing the Brand relationship makes more sense
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasOne
     */
    public function brand()
    {
        return $this->belongsTo(Brand::class, 'brand_id');
    }

    /**
     * Relationship with Seller
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function seller()
    {
        return $this->belongsTo(Member::class, 'seller_id');
    }

    /**
     * Relationship with Buyer
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function buyer()
    {
        return $this->belongsTo(Member::class, 'buyer_id');
    }

    /**
     * Lot Fees
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function fees()
    {
        return $this->hasMany(Fee::class);
    }

    /**
     * Lot Bids
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function bids()
    {
        return $this->hasMany(Bid::class);
    }

    /**
     * Lot Seller Delivery Information
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function seller_delivery(){
        return $this->hasOne(SellerDelivery::class);
    }
    
    /**
     * Lot Seller Payout Information
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function seller_payout(){
        return $this->hasOne(SellerPayout::class);
    }
    
    
    /**
     * Lot Images
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function images()
    {
        return $this->hasMany(\Mtc\Auction\Lot\Image::class)
                    ->orderBy('order');
    }
    
    
    /**
     * Lot default image
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function defaultImage()
    {
        return $this->hasOne(\Mtc\Auction\Lot\Image::class, 'lot_id')
                    ->where('default', 1);
    }

    /**
     * Hover lot Image
     *
     * @return HasOne
     */
    public function hoverImage()
    {
        return $this->hasOne(Image::class)
                    ->where('hover', 1);
    }

    /**
     * Return Latest Bid on Lot
     * 
     * @return \Illuminate\Database\Query\Builder|static
     */
    public function latestBid()
    {
        return $this->hasOne(Bid::class)
                    ->latest()
                    ->latest('id');
    }

    /**
     * Return Latest Bid for a Member
     * 
     * @return Model|false
     */
    public function myLatestBid($member_id)
    {
        return $this->latestBid()
                    ->where('member_id', $member_id)
                    ->first();
    }

    /**
     * Relationship with Custom fields of lot
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function custom()
    {
        return $this->hasOne(Custom::class);
    }

    /**
     * Custom Fields relationship.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function customMultiple()
    {
        return $this->hasMany(CustomMultiple::class);
    }

    /**
     * Lot might have a number of watchers
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function watchers()
    {
        return $this->hasMany(Watchlist::class, 'lot_id');
    }

    /**
     * Get Storage Lots relationship to this Lot.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function storageLots()
    {
        return $this->hasMany(Storage::class);
    }
    
    /**
     * Get Late Fees relationship to this Lot.
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasOne
     */
    public function lateFees()
    {
        return $this->hasOne(LotLatePaymentFee::class);
    }
    
    /**
     * Get Insurance Fees relationship to this Lot.
     *
     * @return \Illuminate\Database\Eloquent\Relations\hasOne
     */
    public function insuranceFees()
    {
        return $this->hasOne(LotInsurance::class);
    }

    /**
     * Relationship with method that will be used for delivering lots to auction site
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    /*public function deliveryMethod()
    {
        return $this->belongsTo(PickupMethod::class, 'delivery_method_id');
    }*/

    /**
     * Do anti-sniping - extend lot lifetime
     */
    public function triggerAntiSnipeExtend()
    {
        $reserve_lot_ids = self::query()
                               ->withBidValue()
                               ->live()
                               ->where('reserve_price', '>', 0)
                               ->havingRaw('`bid_value` < `reserve_price`')
                               ->get()
                               ->pluck('id');

        $query = self::query();

        if (!empty($this->custom->bottle_grouping)) {
            $query->whereHas('custom', function ($query) {
                $query->where('bottle_grouping', $this->custom->bottle_grouping);
            });
        } else {
            $query->where('id', $this->id);
        }

        if ($reserve_lot_ids->count() > 0) {
            $query->orWhereIn('id', $reserve_lot_ids);
        }

        $query->update([
            'ends_at' => Carbon::now()
                               ->addMinutes(config('auction.settings.anti_sniping_extend_minutes')),
        ]);

        //Update overall lot timer
        $auction = Auction::query()->find($this->auction->id);
        $auction->update([
            'ends_at' => Carbon::now()
                               ->addMinutes(config('auction.settings.anti_sniping_extend_minutes')),
        ]);
    }

    /**
     * Scope to filter only live lots
     *
     * @param Builder $query
     *
     * @return Builder
     */
    public function scopeLive(Builder $query)
    {
        return $query->where('status', Status::LIVE)
                     ->where('ends_at', '>', Carbon::now())
                     ->whereNull('buy_out_price');
    }

    
    /**
     * Scope to filter only past lots
     *
     * @param Builder $query
     *
     * @return Builder
     */
    public function scopePast(Builder $query)
    {
        return $query->whereIn('status', Status::$past_statuses)
                     ->where('ends_at', '<', Carbon::now())
                     ->whereNull('buy_out_price');
    }

    /**
     * Scope to filter only future lots
     *
     * @param Builder $query
     *
     * @return Builder
     */
    public function scopeFuture(Builder $query)
    {
        return $query->whereIn('status', Status::$future_statuses)
                     ->where('ends_at', '>', Carbon::now())
                     ->whereNull('buy_out_price');
    }

    /**
     * Scope to filter only buy it now lots
     *
     * @param Builder $query
     *
     * @return Builder
     */
    public function scopeBuyItNow(Builder $query)
    {
        return $query->whereNotNull('buy_out_price')
                     ->whereIn('status', Status::$buy_it_now_statuses);
    }

    /**
     * Set the sort options on query
     *
     * @param Builder $query
     * @param         $sort_by
     *
     * @return Builder
     */
    public function scopeSetListingSort(Builder $query, $sort_by)
    {
        if (empty($sort_by)) {
            $sort_by = 'highest-bids';
        }

        $sort_options = self::getListSortOptions();
        if (array_key_exists($sort_by, $sort_options)) {
            $sort_class = new $sort_options[$sort_by]['query']($query);
            $arguments = $sort_options[$sort_by]['arguments'] ?? [];
            $query = $sort_class->handle(...array_values($arguments));
        }

        return $query;
    }

    /**
     * Filter lots by auction id
     *
     * @param Builder $query
     * @param         $auction_id
     *
     * @return Builder
     */
    public function scopeSetAuctionId(Builder $query, $auction_id)
    {
        if (!empty ($auction_id)) {
            $query->where('auction_id', $auction_id);
        }

        return $query;
    }

    /**
     * Set Status filter for lots
     *
     * @param Builder $query
     * @param         $status
     *
     * @return Builder
     */
    public function scopeSetStatusFilter(Builder $query, $status)
    {
        switch ($status) {
            case "pre_auction":
                return $query->whereIn('status', Status::$pre_auction_statuses);
                break;
            case "live":
                return $query->where('status', Status::LIVE);
                break;
            case "post_auction":
                return $query->whereIn('status', Status::$past_statuses);
                break;
            default:
                return $query;
                break;
        }
    }

    /**
     * Add bid_value column to lot query
     *
     * @param Builder $query
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function scopeWithBidValue(Builder $query, $select_all = true)
    {
        $bid_table = (new Bid)->getTable();
        $lot_table = $this->getTable();

        $sub_query = DB::table($bid_table)
                       ->selectRaw('bid')
                       ->whereRaw($lot_table . '.id = ' . $bid_table . '.lot_id')
                       ->where('deleted_at', null)
                       ->latest()
                       ->latest('id')
                       ->limit(1);

        $sub_query = 'coalesce((' . $sub_query->toSql() . '), 0)';
        $sub_query = 'greatest(' . $sub_query . ', coalesce(`auction_lots`.`starting_price`, 0))';

        if ($select_all) {
            $query->select($lot_table . '.*');
        }

        return $query->selectSub($sub_query, 'bid_value');
    }

    /**
     * Add highest_bidder column to lot query
     *
     * @param Builder $query
     *
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function scopeWithHighestBidder(Builder $query, $select_all = true)
    {
        $bid_table = (new Bid())->getTable();
        $lot_table = $this->getTable();

        $sub_query = DB::table($bid_table)
                       ->selectRaw('member_id')
                       ->whereRaw($lot_table . '.id = ' . $bid_table . '.lot_id')
                       ->where('deleted_at', null)
                       ->latest()
                       ->latest('id')
                       ->limit(1);

        if ($select_all) {
            $query->select($lot_table . '.*');
        }

        return $query->selectSub($sub_query, 'highest_bidder');
    }

    /**
     * Scope to filter lots that should end
     *
     * @param Builder $query
     *
     * @return Builder
     */
    public function scopeShouldEnd(Builder $query)
    {
        return $query->where('status', Status::LIVE)
                     ->where('ends_at', '<=', Carbon::now());
    }

    /**
     * Scope for filtering only viewable (public side) lots
     *
     * @param Builder $query
     *
     * @return mixed
     */
    public function scopeViewable(Builder $query)
    {
        return $query->where('published', 1);
    }

    /**
     * Get the next lot in ID order.
     *
     * @return Lot|null
     */
    public function next()
    {
        if ($this->exists && !empty($this->lot_id)) {
            return self::where(config('auction.lots.identifier'), '>', $this->lot_id)
                       ->orderBy(config('auction.lots.identifier'))
                       ->first();
        }

        return null;
    }

    /**
     * Get the next lot in ID order.
     *
     * @return Lot|null
     */
    public function nextLotInAuction()
    {
        if ($this->exists && !empty($this->lot_id)) {
            return self::where(config('auction.lots.identifier'), '>', $this->lot_id)
                       ->where('auction_id', $this->auction_id)
                       ->where('status', '!=', Status::CANCELLED)
                       ->orderBy(config('auction.lots.identifier'))
                       ->first();
        }

        return null;
    }
    /**
     * Get the next lot in ID order.
     *
     * @return Lot|null
     */
    public function nextLotInAuctionURL()
    {
        if ($this->exists && !empty($this->lot_id)) {
            $lot = self::where(config('auction.lots.identifier'), '>', $this->lot_id)
                       ->where('auction_id', $this->auction_id)
                       ->where('status', '!=', Status::CANCELLED)
                       ->orderBy(config('auction.lots.identifier'))
                       ->first();
            if(!is_null($lot)){
                return $lot->getUrlAttribute();
            }
        }

        return null;
    }
    
    /**
     * Does the lot have a buyer?
     *
     * @return bool
     */
    public function hasBuyer()
    {
        if (!$this->relationLoaded('buyer')) {
            $this->load('buyer');
        }

        return ($this->buyer ? true : false);
    }

    /**
     * Get the previous lot in ID order.
     *
     * @return Lot|null
     */
    public function previous()
    {
        if ($this->exists && !empty($this->lot_id)) {
            return self::where(config('auction.lots.identifier'), '<', $this->lot_id)
                       ->orderBy(config('auction.lots.identifier'), 'desc')
                       ->first();
        }

        return null;
    }

    /**
     * Get the previous lot in ID order.
     *
     * @return Lot|null
     */
    public function previousLotInAuction()
    {
        if ($this->exists && !empty($this->lot_id)) {
            return self::where(config('auction.lots.identifier'), '<', $this->lot_id)
                       ->where('auction_id', $this->auction_id)
                       ->where('status', '!=', Status::CANCELLED)
                       ->orderBy(config('auction.lots.identifier'), 'desc')
                       ->first();
        }

        return null;
    }
    
    /**
     * Get the previous lot in ID order.
     *
     * @return Lot|null
     */
    public function previousLotInAuctionURL()
    {
        if ($this->exists && !empty($this->lot_id)) {
            $lot =  self::where(config('auction.lots.identifier'), '<', $this->lot_id)
                       ->where('auction_id', $this->auction_id)
                       ->where('status', '!=', Status::CANCELLED)
                       ->orderBy(config('auction.lots.identifier'), 'desc')
                       ->first();
            if(!is_null($lot)){
                return $lot->getUrlAttribute();
            }
        }

        return null;
    }

    /**
     * Check if lot is biddable
     *
     * @return bool
     */
    public function isBiddable()
    {
        if ($this->is_buy_it_now) {
            return in_array($this->status, Status::$buy_it_now_statuses);
        }

        return $this->published && $this->status == Status::LIVE && $this->ends_at > Carbon::now();
    }

    /**
     * Get the Lot DB ID based on lot_id used in URL
     *
     * @param $lot_id
     *
     * @return mixed
     */
    public static function decodeLotIdToId($lot_id)
    {
        if (!config('auction.settings.lot_id_to_include_auction_id')) {
            return $lot_id;
        }

        $reference = self::query()
                         ->where(config('auction.lots.identifier'), $lot_id)
                         ->first()->id ?? false;

        if (!$reference) {
            $reference = $lot_id;
        }

        return $reference;
    }

    /**
     * Accessor for lot url
     *
     * @return string
     */
    public function getUrlAttribute()
    {
        //Changed this to make sure there isn't the weird // in urls. Also for locales

        $locale = App::getLocale();
        $site_url = app('url')->to('/');

        if (substr($site_url, -1) == '/') {
            $site_url = substr_replace($site_url, "", -1);
        }

        if ($locale !== 'en') {
            return $site_url . '/' . $locale . '/' . $this->slugify_string("lot-{$this->lot_id}") . '/' . $this->getUrlSafeName();
        } else {
            return $site_url . '/' . $this->slugify_string("lot-{$this->lot_id}") . '/' . $this->slugify_string($this->name);
        }
    }

    /**
     * Accessor for lot whether to show the invoice
     *
     * @return boolean
     */
    public function getShowInvoiceLinkAttribute()
    {
        return !empty($this->buy_order_id);
    }



    /**
     * Can't url encode chinese characters
     * For the sake of SEO I guess, we should re-get the english name of the lot and encode that instead
     *
     * @return string
     */
    public function getUrlSafeName()
    {
        $name = self::where('id', $this->id)->first()->toArray();

        return str_slug($name['name']);
    }

    /**
     * Get the public facing lot ID based on id configuration
     *
     * @return mixed|string
     */
    public function getLotIdAttribute()
    {
        return $this->{config('auction.lots.identifier')};
    }
    
    /**
     * Get the total number of bids on the lot
     *
     * @return mixed|string
     */
    public function getTotalBids()
    {
        return $this->bids->count();
    }
    
    /**
     * Get the total number of watchers on the lot
     *
     * @return mixed|string
     */
    public function getTotalWatchers()
    {
        return $this->watchers->count();
    }

    /**
     * Load the custom fields that should be shown for this lot
     *
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public function prepareCustomFields()
    {
        return collect();
        
        /*$set_id = $this->determineLotCustomFieldSet();

        return CustomFieldSetField::query()
                                  ->with('field.propertyOptions')
                                  ->where('set_id', $set_id)
                                  ->where('auction_lots', 1)
                                  ->get()
                                  ->each(function ($custom_field) {
                                      $custom_field->options = $custom_field->field->propertyOptions->keyBy('name')
                                                                                                    ->map(function ($option) {
                                                                                                        return $option->name;
                                                                                                    });
                                  });*/
    }

    public function requiresShipping()
    {
        return true;
    }

    public function isAvailable(): bool
    {
        if(in_array($this->status, Status::$purchasable_statuses)){
            return true;
        } 
        return false;
    }

    public function getWeight()
    {
        return $this->weight;
    }

    public function getUrl()
    {
        return $this->getUrlAttribute();
    }

    public function getBasketImage()
    {
        return optional($this->defaultImage)->getUrl('mini');
    }

    public function getOriginalPrice()
    {
        /*if($this->vat_on_hammer_price){
            return $this->hammer_price_with_vat + $this->buyer_commission_with_vat;
        }
        return $this->hammer_price + $this->buyer_commission_with_vat;*/


        $price_value = $this->hammer_price + $this->buyer_commission_with_vat;
        if ($this->vat_on_hammer_price) {
            $price_value = $this->hammer_price_with_vat + $this->buyer_commission_with_vat;
        }

        $price = new Price($price_value, $this->taxAttribute());
        //$price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, $key, $price->tax_rate)));
        $price->calculate();

        return $price;
    }

    public function hasDiscountedPrice()
    {
        // TODO: Implement hasDiscountedPrice() method.
    }

    public function getPrice()
    {
        $price_value = $this->hammer_price + $this->buyer_commission_with_vat;
        if ($this->vat_on_hammer_price) {
            $price_value = $this->hammer_price_with_vat + $this->buyer_commission_with_vat;
        }

        $price = new Price($price_value, $this->taxAttribute());
        //$price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, $key, $price->tax_rate)));
        $price->calculate();

        return $price;
    }

    /**
     * Show discount value
     *
     * @return mixed
     */
    public function getPriceExVat()
    {
        return $this->highest_bid + $this->buyer_commission;
    }
    public function taxAttribute()
    {
        if($this->vat_on_hammer_price){
            return ($this->hammer_price_with_vat - $this->hammer_price) + ($this->buyer_commission_with_vat - $this->buyer_commission);
        }
        return $this->buyer_commission_with_vat - $this->buyer_commission;
    }

    public function getName()
    {
        return $this->name;
    }

    public function getSku()
    {
        return $this->getLotIdAttribute();
    }

    public function getStock()
    {
        if(in_array($this->status, Status::$purchasable_statuses)){
            return 1;
        }
        return 0;
    }

    public function getMaxQuantity()
    {
        return 1;
    }

    public function getAttributeFields(): array
    {
        // TODO: Implement getAttributeFields() method.
        return [];
    }

    public function scopeSearchByTerm(\Illuminate\Database\Eloquent\Builder $query, string $term)
    {
        
    }

    public function getSearchNameAttribute(): string
    {
        return collect([
            $this->name,
            $this->{config('auction.lots.identifier')},
            $this->price
        ])->filter()
          ->implode(', ');
    }

    public function slugify_string($string){
        return strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-',$string)));
    }



    public function incrementViewCounter()
    {
        if (is_null($this->last_viewed_at)) {
            $this->num_views = 1;
        } else {
            $this->attributes['num_views']++;
        }
        $this->last_viewed_at = Carbon::now();
        $this->save();
    }
    
    public static function getAdditionalDisplayFields(){
        return [];

        return config('auction.lots.display_fields');
    }

    /**
     * Check if user can bid on this lot
     *
     * @param Member $member or AuctionMember $member
     *
     * @return bool
     */
    public function canUserBidOnThisLot(AuctionMember $member, $check_member = true)
    {
        $this->not_allowed_to_bid = false;
        if ($check_member && !$member->can_bid && (config('auction.settings.registration_fee_for_bidders') && (is_null($member->auction_fields) || !$member->auction_fields->buyer_membership_fee_paid)) ) {
            $this->not_allowed_to_bid = __('auction::lots.bidding.must_pay_registration_fee');
        }

        $this->is_live = $this->isBiddable();
        $is_users_lot = $member->id == $this->seller_id;

        if ($is_users_lot) {
            $this->not_allowed_to_bid = __('auction::lots.bidding.own_lots');
        }

        if (!$this->isBiddable()) {
            $this->not_allowed_to_bid = __('auction::lots.bidding.not_biddable_lot');
        }

        if ($this->is_buy_it_now) {
            $this->is_live = in_array($this->status, Status::$buy_it_now_statuses);
            $this->not_allowed_to_bid = false;
        }

        return $this->is_live
               && !$this->not_allowed_to_bid
               && (!$check_member || ($check_member && $member->exists))
               && !$is_users_lot;
    }

    /**
     * reserve_met attribute for quick checks
     *
     * @return bool
     */
    public function getReserveMetAttribute()
    {
        return $this->reserve_price <= ($this->latestBid->bid ?? 0);
    }

    /**
     * Descriptor of current bid for lot
     *
     * @return string
     */
    public function getBidDescriptorValueAttribute()
    {
        if ($this->status == Status::NOT_SOLD) {
            return 'highest_bid';
        } elseif (in_array($this->status, Status::$winning_statuses)) {
            return 'winning_bid';
        } elseif ($this->bids()->count() > 0) {
            return 'current_bid';
        }

        return 'starting_bid';
    }

    /**
     * Descriptor of current bid for lot
     *
     * @return string
     */
    public function getBidDescriptorTextAttribute()
    {
        return __('auction::lots.bidding.' . $this->bid_descriptor_value);
    }

    public function getNextBidIncrementAttribute()
    {
        return Rule::whereRaw('? BETWEEN `min` AND `max`')
                   ->addBinding($this->latestBid->bid ?? $this->starting_price ?? 0)
                   ->first()->step_size;
    }

    /**
     * Auto-magically decode lot payout details to $lot->payout_data
     *
     * @return array
     */
    public function getPayoutDataAttribute()
    {
        return empty($this->payout_details) ? [] : json_decode(decrypt($this->payout_details), true);
    }

    /**
     * Auto-magically decode lot courier details to $lot->courier_data
     *
     * @return array
     */
    public function getCourierDataAttribute()
    {
        return empty($this->courier_details) ? [] : json_decode(decrypt($this->courier_details), true);
    }

    /**
     * Set the lot's seller commission.
     *
     * @param string $value
     *
     * @return void
     */
    public function setSellerCommissionAttribute($value)
    {
        $this->attributes['seller_commission'] = ($value === '') ? null : $value;
    }

    /**
     * Set the lot's seller commission limit.
     *
     * @param string $value
     *
     * @return void
     */
    public function setSellerCommissionLimitAttribute($value)
    {
        $this->attributes['seller_commission_limit'] = ($value === '') ? null : $value;
    }

    /**
     * Get the time_remaining value
     * This gives seconds until Lot timer expires.
     *
     * @return int
     */
    public function getTimeRemainingAttribute()
    {
        
        if(is_null($this->auction)){
            return 0;
        }
        
        $start_datetime = $this->auction->starts_at;
        $end_datetime = $this->auction->ends_at;

        if (Carbon::now() < $start_datetime) {
            return Carbon::now()
                         ->diffInSeconds($start_datetime);
        }

        return Carbon::now()
                     ->diffInSeconds($end_datetime, false);
    }

    /*public function getStartingPriceAttribute()
    {
        return $this->attributes['starting_price'] ?? $this->getMinimumPriceByLaw();
    }*/

    public function setBuyItNowPriceAttribute($value)
    {
        $this->attributes['buy_it_now_price'] = empty($value) ? null : $value;
    }

    public function getIsBuyItNowAttribute()
    {
        return !is_null($this->buy_it_now_price);
    }

    public function getNumViewsAttribute()
    {
        if ($this->attributes['num_views'] < 8) {
            return rand(4, 8);
        }

        return $this->attributes['num_views'];
    }

    /**
     * Re-list a lot
     */
    public function relist()
    {
        $this->status = Status::RELISTED;
        $this->save();

        $this->cloneToRelist(false);
    }
    
    public function cloneToRelist($create_additional_fees = false, $use_next_available_auction = false, $record_relist_history = true, $remove_buyer = false, $remove_seller = false){
        if ($use_next_available_auction) {
            $auction = Auction::nextAvailableAuction();
        } else {
            $auction = Auction::nextSubmittableAuction();
        }
        
        $new_lot = $this->replicate([
            'relisted_from_lot_id',
            'auction_id',
            'status',
            'reference',
            'buyer_id',
            'seller_id',
            'highest_bid',
            'hammer_price',
            'hammer_price_with_vat',
            'buyer_commission',
            'buyer_commission_with_vat',
            'seller_commission',
            'seller_commission_with_vat',
            'num_views',
            'lot_condition',
        ])->fill([
            'relisted_from_lot_id' => ($record_relist_history) ? null : $this->id,
            'buyer_id' => ($remove_buyer) ? null : $this->buyer_id,
            'seller_id' => ($remove_seller) ? null : $this->seller_id,
            'auction_id' => (is_null($auction)) ? null : $auction->id,
            'status' => Status::REGISTERED
        ]);

        $new_lot->save();
        
        $this->images->each(function (Image $image) use ($new_lot) {
            $new_lot->images()
                ->create($image->getAttributes());
        });

        $new_lot->custom()
            ->create($this->custom->getAttributes() ?? []);

        if ($create_additional_fees) {
            //TODO - Fees

            if (empty(floatval($lot->reserve_price))) {
                // make sure there is only one reserve price fee
            } else {
                //clone all the old fees
            }
        }
    }

    /**
     * Get Sort options for lots
     *
     * @return array
     */
    public static function getListSortOptions()
    {
        return [
            'highest-bids' => [
                'name' => __('auction::lots.filter.sort.highest_bids'),
                'query' => SortByPrice::class,
            ],
            'lowest-bids' => [
                'name' => __('auction::lots.filter.sort.lowest_bids'),
                'query' => SortByPrice::class,
                'arguments' => [
                    'direction' => 'asc',
                ],
            ],
            'alphabetic-az' => [
                'name' => __('auction::lots.filter.sort.alphabetic_az'),
                'query' => SortByName::class,
            ],
            'alphabetic-za' => [
                'name' => __('auction::lots.filter.sort.alphabetic_za'),
                'query' => SortByName::class,
                'arguments' => [
                    'direction' => 'desc',
                ],
            ],
            'lot_id' => [
                'name' => __('auction::lots.filter.sort.lot_id'),
                'query' => SortByLotId::class,
            ],
        ];
    }
    
    /**
     * Get Sort options for lots
     *
     * @return array
     */
    public static function getListSortOptionsFlat()
    {
        return [
            'highest-bids' => __('auction::lots.filter.sort.highest_bids'),
            'lowest-bids' =>  __('auction::lots.filter.sort.lowest_bids'),
            'alphabetic-az' =>  __('auction::lots.filter.sort.alphabetic_az'),
            'alphabetic-za' =>  __('auction::lots.filter.sort.alphabetic_za'),
            'lot_id' =>  __('auction::lots.filter.sort.lot_id'),
        ];
    }



    public static function removeHoldInStorageDeliveryOptionIfNoLotsAreInBasket(SetDeliveryForBasket $setDeliveryForBasket){
        $lots_are_actually_in_basket = $setDeliveryForBasket->getBasket()->items->where('purchasable_type', 'Mtc\Auction\Lot')->count();
        if($lots_are_actually_in_basket === 0){
            $setDeliveryForBasket->getBasket()->surcharges()->where('surcharge_type', 'delivery')->where('surcharge_id', config('auction.storage.delivery_id'))->delete();
            $setDeliveryForBasket->getBasketRepository()->setErrors(['delivery_method' => 'You cannot select Storage as this option is only available when paying for lots']);
        }
    }
    
    public function getInitialBidValue(){
        $current_bid = $this->latestBid->bid ?? 0;
        
        if($current_bid < $this->starting_price){
            $current_bid = $this->starting_price;
        }
        
        $relevant_rule = Rule::where('min', '<=', $current_bid)->where('max', '>=', $current_bid)->first();

        if(is_null($relevant_rule)){
            return $current_bid;
        } else {
            return $current_bid + $relevant_rule->step_size;
        }
    }
}