<?php

namespace Mtc\Auction\Lot;

use Mtc\Auction\Lot\Status;
use Mtc\Auction\Lot\ProcessQueue;

trait EndProcess
{
    /**
     * @return mixed
     */
    public static function endOverDueLots()
    {
        return self::query()
                   ->shouldEnd()
                   ->limit(50)
                   ->get()
                   ->each(function (self $lot) {
                       $lot->endAuctionForLot();
                   });
    }

    /**
     * Check if auction has ended and there are not any lots live
     * or pending processing
     *
     * @return bool
     */
    public static function canSendProcessingEmails()
    {
        return self::query()->live()->count() == 0
               && self::query()->shouldEnd()->count() == 0;
    }

    /**
     * End Auction for Lot
     */
    protected function endAuctionForLot()
    {
        $this->highest_bid = $this->latestBid->bid ?? 0;

        if ($this->wasLotSold($this->highest_bid)) {
            $this->markAsSold();
            // Add this to queue for buyer as it was sold
            ProcessQueue::query()
                        ->create([
                            'lot_id' => $this->id,
                            'member_id' => $this->buyer_id,
                            'action' => 'buyer_process'
                        ]);
        } else {
            $this->markAsNotSold();
        }

        // Add this to queue for sellers
        ProcessQueue::query()
                    ->create([
                        'lot_id' => $this->id,
                        'member_id' => $this->seller_id,
                        'action' => 'seller_process'
                    ]);

        /*if (WATCHLIST_ENABLED === true) {
            $this->watchlistForEndedLot();
        }*/
    }

    /**
     * Set watchlist notifications for lot that has ended
     */
    protected function watchlistForEndedLot()
    {
        /*Watchlist::query()
                 ->where('lot_id', $this->id)
                 ->get()
                 ->each(function ($watchlist) {
                     // Add this to queue for watchers
                     ProcessQueue::query()
                                 ->create([
                                     'lot_id' => $this->id,
                                     'member_id' => $watchlist->member_id,
                                     'action' => 'watchlist_process',
                                 ]);
                 });*/
    }


    /**
     * Check if lot was sold based on the received bid
     *
     * @param $highest_bid
     * @return bool
     */
    public function wasLotSold($highest_bid)
    {
        if (!$this->hasBidsPlaced()) {
            return false;
        }

        // No reserve price = sold
        if ($this->reserve_price == 0) {
            return true;
        }

        // Lastly - sold id bid over reserve price
        return $highest_bid >= $this->reserve_price;

    }

    /**
     * Check if lot has bids
     *
     * @return bool
     */
    public function hasBidsPlaced()
    {
        return $this->bids()->exists();
    }

    /**
     * Mark as Sold
     */
    public function markAsSold()
    {
        // Buyer commission tiers
        if (floatval(config('auction.default_buyer_commission_percentage')) === floatval($this->buyer_commission_percentage)) {
            $buyer_commission_tiers = config('auction.buyer_commission_tiers');
            if (is_array($buyer_commission_tiers)) {
                foreach ($buyer_commission_tiers as $buyer_commission_tier) {
                    if ($this->highest_bid >= $buyer_commission_tier['min_highest_bid'] && $this->highest_bid <= $buyer_commission_tier['max_highest_bid']) {
                        $this->buyer_commission_percentage = $buyer_commission_tier['buyer_commission_percentage'];
                        break;
                    }
                }
            }
        }

        $this->status = Status::SOLD;
        $this->buyer_id = $this->latestBid->member_id;
        
        // Seperating the Hammer Price from the Highest Bid
        $this->hammer_price = $this->highest_bid;
        $this->hammer_price_with_vat = $this->hammer_price * (1 + VAT_RATE * 0.01);
        
        $this->buyer_commission = $this->hammer_price * $this->buyer_commission_percentage * 0.01;
        $this->buyer_commission_with_vat = $this->buyer_commission * (1 + VAT_RATE * 0.01);
        if (!empty($this->seller_commission_limit) && $this->seller_commission_limit > $this->hammer_price) {
            $this->seller_commission = $this->seller_commission_with_vat = 0;
        } else {
            $this->seller_commission = $this->hammer_price * $this->seller_commission_percentage * 0.01;
            $this->seller_commission_with_vat = $this->seller_commission * (1 + VAT_RATE * 0.01);
        }

        $this->save();

        /*try {
            $params = [
                'member_id' => $this->buyer_id,
                'url' => '/members/won.php',
            ];
            Notification::add($params);
        } catch (\Exception $exception) {

        }*/
    }

    /**
     * Mark as Not sold
     */
    public function markAsNotSold()
    {
        $this->status = Status::NOT_SOLD;
        $this->save();
        
        if($this->relisting_enabled) {
            $this->relist();
        }

        /*try {

            $params = [
                'member_id' => $this->seller_id,
                'url' => '/members/my-lots/past',
                'message' => "Lot auction ended, Lot #" . $this->id . ": " . $this->name . " was not sold",
            ];
            Notification::add($params);
        } catch (\Exception $exception) {

        }*/
    }
}