<?php

namespace Mtc\Auction\Lot;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Storage;
use Mtc\Core\Facades\Media;

class Image extends Model
{

    const DEFAULT_IMAGE_PATH = '';
    
    /**
     * Classes associated table
     *
     * @var string $table
     */
    protected $table = 'auction_lot_images';

    /**
     * Mass assignable attributes
     *
     * @var array $fillable
     */
    protected $fillable = [
        'lot_id',
        'name',
        'default',
        'hover',
        'zoom',
        'order',
    ];

    /**
     * Extend model booting
     */
    protected static function boot()
    {
        parent::boot();

        self::addGlobalScope(function ($query) {
            return $query->orderBy('order', 'ASC');
        });

        self::deleted(fn(self $image) => $image->deleteImageFiles());
    }

    /**
     * Scope - isDefault()
     * Find the default image for item
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeIsDefault($query)
    {
        return $query->where('default', 1);
    }

    /**
     * Scope - isZoom()
     * Find the zoom image for item
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeIsZoom($query)
    {
        return $query->where('zoom', 1);
    }

    /**
     * Get image url for a specific size
     *
     * @param string $size
     * @return string
     */
    public function getUrl($size = 'thumbs')
    {
        $image_folders = App::make('image_folders');
        return '/' . $image_folders['lot_images'][$size]['path'] . '/' . $this->name;
    }
    
    /**
     * Alias for the getUrl function
     *
     * @param string $size
     * @return string
     */
    public function imageUrl($size = 'thumbs')
    {
        return $this->getUrl($size);
    }

    /**
     * Remove image files from storage
     *
     * @return void
     */
    public function deleteImageFiles()
    {
        collect(Media::getFolders('lot_images'))
            ->map(fn ($folder) => str_replace('uploads/', '', $folder['path']) . '/' . $this->name)
            ->each(fn ($image_path) => Storage::disk('uploads')->delete($image_path));
    }
}