<?php

namespace Mtc\Auction\Lot;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\Auction\Lot;
use Mtc\Auction\Lot\Status;
use Mtc\Auction\AuctionMember;

/**
 * Lot Process Queue
 *
 * @category Auction
 * @package  Mtc_Auction
 * @author   mtc. Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
Class ProcessQueue extends Model
{
    /**
     * @var String Classes associated table
     */
    protected $table = 'auction_lot_post_live_process_queue';

    /**
     * @var array $fillable mass assignable attributes
     */
    protected $fillable = [
        'lot_id',
        'action',
        'member_id'
    ];

    /**
     * Relationship with a lot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function lot()
    {
        return $this->belongsTo(Lot::class, 'lot_id');
    }

    /**
     * Relationship with a lot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(AuctionMember::class, 'member_id');
    }

    /**
     * Get outstanding tasks for a specific queue
     *
     * @param $action
     * @return Collection
     */
    protected static function getByAction($action)
    {
        return self::query()
            ->with([
                'lot',
                'member'
            ])
            ->where('action', $action)
            ->get()
            ->groupBy('member_id')
            ->take(50);
    }

    /**
     * Process entries in queue
     *
     * @param $action
     */
    public static function processBuyer()
    {
        $twig = App::make('twig');
        $lot_groups = self::getByAction('buyer_process');

        $lot_groups->each(function ($lot_queue) use ($twig) {
            $member = $lot_queue->first()->member;
            $lots = $lot_queue->map(function ($queue_item) {
                return $queue_item->lot;
            });

            $subject = config('app.name') . " - Auction Ended, You've won!";
            $content = $twig->render('Auction/emails/won.twig', [
                'lots' => $lots,
                'member' => $member,
                'purchase_url' => config('app.url') . '/purchase-lots',
                'bank_transfer_details' => config('auction.bank_transfer_details'),
            ]);
            email($member->email, $subject, $content);

            $lot_queue->each(function ($queue_item) {
                $queue_item->delete();
            });

        });
    }

    /**
     * Process entries in queue
     *
     * @param $action
     */
    public static function processSeller($include_trustpilot = true)
    {
        $twig = App::make('twig');
        $lot_groups = self::getByAction('seller_process');

        $lot_groups->each(function ($lot_queue) use ($twig, $include_trustpilot) {

            $lot = $lot_queue->first();
            
            if(is_null($lot->seller_id)){
                return;
            }
            
            $member = AuctionMember::query()->find($lot->seller_id);
            
            $lots = $lot_queue->map(function ($queue_item) {
                return $queue_item->lot;
            });

            $lots_with_no_payout_details = $lots
                ->where('payout_details', '')
                ->isNotEmpty();

            $subject = config('app.name') . " - Auction Ended";
            $content = $twig->render('Auction/emails/seller_auction_end.twig', [
                'lots' => $lots,
                'unsold_lots' => $lots->where('status', Status::NOT_SOLD),
                'member' => $member,
                'lots_with_no_payout_details' => $lots_with_no_payout_details,
                'lot_statuses' => Lot\Status::$lot_statuses,
            ]);
            
            email($member->email, $subject, $content);

            //Send email to Trustpilot
            if($include_trustpilot && !empty(config('auction.trustpilot_bcc_email'))){
                email(config('auction.trustpilot_bcc_email'), $subject, $content);
            }

            $lot_queue->each(function ($queue_item) {
                $queue_item->delete();
            });

        });


    }

    /**
     * Process entries in queue
     *
     * @param $action
     */
    public static function processWatchlist()
    {
        $twig = App::make('twig');
        $lot_groups = self::getByAction('watchlist_process');

        $lot_groups->each(function ($lot_queue) use ($twig) {

            $member = $lot_queue->first()->member;
            $lots = $lot_queue->map(function ($queue_item) {
                return $queue_item->lot;
            });

            $subject = "Auction for lots in your watchlist has ended on " . config('app.name');
            $content = $twig->render('Auction/emails/watchlist_end.twig', [
                'lots' => $lots,
                'member' => $member,
                'sold_status' => Lot\Status::$winning_statuses,
            ]);

            email($member->email, $subject, $content);

            $lot_queue->each(function ($queue_item) {
                $queue_item->delete();
            });

        });


    }
}
