<?php

namespace Mtc\Auction\Lot;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Mtc\Auction\AuctionMember;
use Mtc\Basket\Contracts\Purchasable;
use Mtc\Members\Member;
use Mtc\Money\HasPrices;
use Mtc\Money\Price;
use Mtc\Orders\Events\OrderStatusChange;
use Mtc\Orders\Order;
use Mtc\Auction\Lot\Status;
use Mtc\Auction\Lot;
use Mtc\Auction\Auction;
use Mtc\Orders\Status as OrderStatus;
use Mtc\Auction\Lot\StorageLocation;
use Mtc\ShippingManager\Events\SetDeliveryForBasket;
use Mtc\ShippingManager\Listeners\ProvideDeliveryMethodsToBasket;

use function PHPUnit\Framework\isNull;

class Storage extends Model implements Purchasable
{
    use SoftDeletes, HasPrices;

    /**
     * Table name
     *
     * @var string $table
     */
    protected $table = 'auction_storage_lots';

    /**
     * Mass assignable attributes
     *
     * @var array $fillable
     */
    protected $fillable = [
        'auction_id',
        'lot_id',
        'member_id',
        'storage_expires_at',
        'location_id',
    ];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $casts = [
        'storage_expires_at' => 'datetime',
    ];

    /**
     * Relationship with Auction
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function auction()
    {
        return $this->belongsTo(Auction::class);
    }

    /**
     * Relationship with Lot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function lot()
    {
        return $this->belongsTo(Lot::class);
    }

    /**
     * Relationship with member
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class, 'member_id');
    }
    
    
    public function auctionMember()
    {
        return AuctionMember::mutateMemberObject($this->member);
    }

    /**
     * Relationship with Storage Locations
     * 
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function location()
    {
        return $this->hasOne(StorageLocation::class);
    }

    /**
     * ->time_stored_for
     *
     * @return string
     */
    public function getTimeStoredForAttribute()
    {
        if ($this->created_at instanceof Carbon) {
            return $this->created_at->diffForHumans();
        }

        return '';
    }

    /**
     * @param OrderStatusChange $event
     *
     * @return void
     */
    public static function processOrderStatusChange(OrderStatusChange $event){
        $order_status = OrderStatus::query()->find($event->getStatus());
        /**
         * TODO - Logic
         * 
         * Storage in the original template was very specific and other clients were just told how it works
         * Originally it worked under the assumption that the lots were physically stored in a location onsite
         * Like "locker 2" and relied of order statuses to also reflect this
         * 
         * Will need to think out more so how this would work when not all clients can do this
         */
        
        
    }
    
    public static function removeHoldInStorageDeliveryOptionIfStoredLotsAreInBasket(SetDeliveryForBasket $setDeliveryForBasket){
        $storage_delivery_selected = $setDeliveryForBasket->getBasket()->surcharges()->where('surcharge_type', 'delivery')->where('surcharge_id', config('auction.storage.delivery_id'))->first();
        $storage_lots_in_basket = $setDeliveryForBasket->getBasket()->items->where('purchasable_type', 'Mtc\Auction\Lot\Storage')->first();
        if($storage_delivery_selected && $storage_lots_in_basket){
            //The user has attempted to select hold in storage while they have a lot shipment in their basket
            $setDeliveryForBasket->getBasket()->surcharges()->where('surcharge_type', 'delivery')->where('surcharge_id', config('auction.storage.delivery_id'))->delete();
            $setDeliveryForBasket->getBasketRepository()->setErrors(['delivery_method' => 'You cannot select Storage as you have stored lots already in your basket']);
        }
    }

    public function isAvailable(): bool{
        return true;
    }

    public function getName()
    {
        $today = new Carbon('today 00:00');
        $name  = $this->lot->name . ' : Shipment';
        if ($today > $this->storage_expires_at) {
            $name .= ' : LATE STORAGE FEES';
        }

        return $name;
    }
    
    public function getWeight(){
        return $this->lot->weight ?? 0;
    }
    
    public function requiresShipping()
    {
        return true;
    }

    public function getUrl()
    {
        return '';
    }

    public function getBasketImage()
    {
        return optional($this->lot->defaultImage)->getUrl('mini');
    }

    public function getOriginalPrice()
    {
        // TODO: Implement getOriginalPrice() method.

        $price_value = 0;

        $today = new Carbon('today 00:00');

        if ($today > $this->storage_expires_at) {

            $price_value = $today->diffInDays($this->storage_expires_at) * config('auction.storage.late_fee_per_day');

        }

        $price = new Price($price_value, 0);
        //$price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, $key, $price->tax_rate)));
        $price->calculate();

        return $price;
    }

    public function hasDiscountedPrice()
    {
        // TODO: Implement hasDiscountedPrice() method.
    }

    public function getPrice()
    {
        $price_value = 0;

        $today = new Carbon('today 00:00');

        if ($today > $this->storage_expires_at) {
            
            $price_value = $today->diffInDays($this->storage_expires_at) * config('auction.storage.late_fee_per_day');
            
        }
        
        $price = new Price($price_value, 0);
        //$price->setModifiers(Event::dispatch(new LoadPriceModifiers($this, $key, $price->tax_rate)));
        $price->calculate();

        return $price;
    }

    public function getSku()
    {
        return 'LOT-' . $this->lot->getLotIdAttribute() . '-SHIPMENT';
    }

    public function getStock()
    {
        // TODO: Implement getStock() method.
        if($this->trashed()){
            return 0;
        }
        return 1;
    }

    public function getMaxQuantity()
    {
        return 1;
    }

    public function getAttributeFields(): array
    {
        // TODO: Implement getAttributeFields() method.
        return [];
    }

    public function scopeSearchByTerm(Builder $query, string $term)
    {
        // TODO: Implement scopeSearchByTerm() method.
        return false;
    }

    public function getSearchNameAttribute(): string
    {
        // TODO: Implement getSearchNameAttribute() method.
        return '';
    }
}