<?php

namespace Mtc\Auction\Observers;

use Mtc\Auction\Auction;
use Mtc\Auction\Lot;

/**
 * Class LotCreating Event
 *
 * @package Mtc\Modules\Auction\Classes\Events
 */
class LotObserver
{
    /**
     * Listen to the Lot creating event.
     *
     * @param Lot $lot
     */
    public function created(Lot $lot)
    {
        $this->setReference($lot);
        $this->setCommissionDefaults($lot);
        $this->reportDirtyFields($lot);
        $this->removeLotFromStorage($lot);
    }
    
    /**
     * Listen to the Lot updating event.
     *
     * @param Lot $lot
     */
    public function updated(Lot $lot)
    {
        $this->setReference($lot);
        $this->setCommissionDefaults($lot);
        $this->reportDirtyFields($lot);
        $this->removeLotFromStorage($lot);
    }

    /**
     * Set the lot reference number based on the auction ID.
     *
     * @param Lot $lot
     */
    private function setReference(Lot $lot)
    {
        if (
            !$lot->isDirty('auction_id') ||
            empty($lot->auction_id)
        ) {
            return;
        }

        $generate_new_lot_reference = true;

        if(isset($lot->reference) && is_numeric($lot->reference)){
            // We should check if the lot reference is currently in use:
            $preexisting_lots = Lot::where('reference', $lot->reference)->where('id', '!=', $lot->id)
                                   ->orderBy('reference', 'desc');

            if(count($preexisting_lots) == 0){
                $generate_new_lot_reference = false;
            }
        }

        if($generate_new_lot_reference === true) {
            $auctions = Auction::orderBy('id')
                               ->pluck('id');

            if (($auction_number = $auctions->search($lot->auction_id)) !== false) {
                $auction_number++;
            }

            if (empty($auction_number)) {
                $lot->reference = null;
            } else {
                if (($last_lot = Lot::where('auction_id', $lot->auction_id)
                                    ->orderBy('reference', 'desc')
                                    ->first())->exists) {
                    $lot_number = intval(substr($last_lot->reference, strlen(str_pad($auction_number, 2, '0', STR_PAD_LEFT))))
                                  + 1;
                } else {
                    $lot_number = 0;
                }

                do {
                    $lot->reference = str_pad($auction_number, 2, '0', STR_PAD_LEFT)
                                      . str_pad($lot_number++, 4, '0', STR_PAD_LEFT);
                } while (Lot::where('reference', $lot->reference)
                            ->exists());
            }
        }
    }

    /**
     * Set the lot commission defaults.
     *
     * @param Lot $lot
     */
    private function setCommissionDefaults(Lot $lot)
    {
        if (is_null($lot->buyer_commission_percentage) && !is_null(config('auction.default_buyer_commission_percentage'))) {
            $lot->buyer_commission_percentage = config('auction.default_buyer_commission_percentage');
        }

        if (is_null($lot->seller_commission_percentage) && !is_null(config('auction.default_seller_commission_percentage'))) {
            $lot->seller_commission_percentage = config('auction.default_seller_commission_percentage');
        }

        if (is_null($lot->seller_commission_limit) && !is_null(config('auction.default_seller_commission_limit'))) {
            $lot->seller_commission_limit = config('auction.default_seller_commission_limit');
        }
    }

    /**
     * Send email to client for changes made on lots.
     *
     * @param Lot $lot
     */
    private function reportDirtyFields(Lot $lot)
    {
        if (!$lot->exists || !defined('AUCTION_LOT_EDITS_EMAIL') || empty(AUCTION_LOT_EDITS_EMAIL)) {
            return;
        }

        $new_attributes = array_diff_key($lot->getDirty(), array_flip($lot->hidden_from_lot_edit_emails));
        $old_attributes = array_intersect_key($lot->getOriginal(), $new_attributes);

        if (!empty($new_attributes)) {
            $content = template('Auction/emails/client_report_lot_update.twig', [
                'lot' => $lot,
                'old_attributes' => $old_attributes,
                'new_attributes' => $new_attributes,
            ]);

            email(config('auction.settings.manager_email'), config('app.name') . ' - Lot #' . $lot->lot_id . ' Updated', $content);
        }
    }


    private function removeLotFromStorage(Lot $lot)
    {
        if (intval($lot->status) === Lot\Status::SHIPPED) {
            Lot\Storage::query()
                   ->where('lot_id', $lot->id)
                   ->delete();
        }
    }

}