<?php

namespace Mtc\Auction\Providers;


use Illuminate\Foundation\Support\Providers\EventServiceProvider;
use Illuminate\Support\Facades\App;
use Mtc\Auction\Auction;
use Mtc\Auction\AuctionMember;
use Mtc\Auction\AuctionMembershipFee;
use Mtc\Auction\Contracts\BidNotifier;
use Mtc\Auction\Events\BidUpdate;
use Mtc\Auction\Http\Response\Lotlist;
use Mtc\Auction\Listeners\AddLotInsuranceFormToBasket;
use Mtc\Auction\Listeners\CacheLotBid;
use Mtc\Auction\Listeners\GetShippingBreakdownTemplate;
use Mtc\Auction\Listeners\GetUpcomingAuctions;
use Mtc\Auction\Listeners\renderAuctionFooterJS;
use Mtc\Auction\Lot;
use Illuminate\Support\Facades\Event;
use Carbon\Carbon;
use Mtc\Auction\Lot\ProcessQueue;
use Mtc\Auction\Observers\LotObserver;
use Mtc\Basket\Events\LoadBasketExtensions;
use Mtc\Core\Currency;
use Mtc\Core\Facades\Media;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Modules\Members\Classes\Auth;
use Mtc\Orders\Events\OrderPaid;
use Mtc\Orders\Events\OrderStatusChange;
use Mtc\ShippingManager\Events\SetDeliveryForBasket;
use Mtc\ShippingManager\Listeners\ProvideDeliveryMethodsToBasket;

class AuctionServiceProvider extends EventServiceProvider
{
    protected $listen = [
        LoadBasketExtensions::class => [
            AddLotInsuranceFormToBasket::class
        ],
    ];

    /**
     * Bootstrap the application services.
     */
    public function boot()
    {
        parent::boot();

        Media::addPath(\dirname(__DIR__, 2) . '/images/image_folders.php');

        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'auctions');

        if ($this->app->runningInConsole()) {
            $this->loadMigrationsFrom(dirname(__DIR__, 2) . '/database/migrations');
            $this->bootResourcePublish();
        } else {
            $this->bootTwigDependencies();
        }
        
        GetUpcomingAuctions::handle();
        GetShippingBreakdownTemplate::handle();
        renderAuctionFooterJS::handle();

        Lot::observe(LotObserver::class);

        app('twig')->addFunction(new \Twig_SimpleFunction('featured_lots', function () {
            $live_auction = Auction::currentLiveAuction();
            $past         = true;
            if (!is_null($live_auction) && $live_auction->id > 0) {
                $past = false;
                $featured_lots = Lot::query()
                                    ->where('is_featured', 1)
                                    ->where('auction_id', $live_auction->id)
                                    ->whereDate('ends_at', '>', Carbon::now())
                                    ->get();
            } else {
                $featured_lots = Lot::query()
                                    ->where('is_featured', 1)
                                    ->whereDate('ends_at', '<=', Carbon::now())
                                    ->get();
            }
            if ($featured_lots->count() > 0) {
                $featured_lot_list = (new Lotlist($featured_lots))->handle(false);
                echo app('twig')->render('Auction/lot/featured_lots.twig', [
                    'featured_lots' => $featured_lot_list,
                    'past'          => $past,
                ]);
            }
        }));

    }

    protected $commands = [
        'Mtc\Auction\Commands\EveryMinute',
        'Mtc\Auction\Commands\Daily',
    ];

    /**
     * Register requirements for Auction module
     */
    public function register()
    {
        // Automatically apply the package configuration
        $this->mergeConfigFrom(\dirname(__DIR__, 2) . '/config/auction.php', 'auction');
        $this->mergeConfigFrom(\dirname(__DIR__, 2) . '/config/basket.php', 'basket');
        $this->mergeConfigFrom(\dirname(__DIR__, 2) . '/config/shipping_manager.php', 'shipping_manager');
        //$this->mergeConfigFrom(\dirname(__DIR__, 2) . '/config/auction_images.php', 'auction_images');

        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'auction');

        $this->app->register(config('auction.routing'));

        self::bindBidNotifier();

        //Events and Hooks

        // Handle the bid update process
        Event::listen(BidUpdate::class, function (BidUpdate $bid_update) {
            /** @var BidNotifier $bid_notifier */
            $bid_notifier = App::make(BidNotifier::class);
            $bid_notifier->handle($bid_update);
        });


        Event::listen(OrderStatusChange::class, Lot\Storage::class . '@processOrderStatusChange');

        Event::listen(OrderPaid::class, function (OrderPaid $order_paid) {
            Lot::processOrderPayment($order_paid);
            Lot\LotLatePaymentFee::processOrderPayment($order_paid);
            AuctionMember::processOrderPaymentForAuction($order_paid);
        });

        Event::listen(SetDeliveryForBasket::class, function (SetDeliveryForBasket $setDeliveryForBasket) {
            Lot\Storage::removeHoldInStorageDeliveryOptionIfStoredLotsAreInBasket($setDeliveryForBasket);
            Lot::removeHoldInStorageDeliveryOptionIfNoLotsAreInBasket($setDeliveryForBasket);
            Lot\LotLatePaymentFee::removeAnyNoLongerApplicableLateFeesOnDeliveryUpdate($setDeliveryForBasket);
            //AuctionMember::processOrderPaymentForAuction($order_paid);
        });
        
        Event::listen(LoadBasketExtensions::class, function (LoadBasketExtensions $loadBasketExtensions) {
            Lot\LotLatePaymentFee::removeAnyNoLongerApplicableLateFees($loadBasketExtensions);
        });

        /* TODO - The old hooks that ran when the items in the basket were loaded */

        $this->commands($this->commands);
    }

    /**
     * Boot twig functionality
     */
    public function bootTwigDependencies()
    {
        app('twig_loader')->addPath(dirname(__DIR__, 2) . '/resources/templates');
    }

    /**
     * Register resources that get published
     */
    protected function bootResourcePublish()
    {
        $this->publishes([
            dirname(__DIR__, 2) . '/config/' => config_path(),
        ], 'config');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/js/' => resource_path('js/vendor'),
        ], 'assets');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/less/' => resource_path('less/vendor'),
        ], 'assets');
    }

    /**
     * Bind the bid notification
     */
    protected static function bindBidNotifier()
    {
        // Fallback to polling cached bid values
        App::bind(BidNotifier::class, function () {
            return new CacheLotBid();
        });
    }

    public static function auctionEndtest()
    {
        $auction = Auction::currentLiveAuction();

        // Delay auction end for 15 minutes
        if ($auction->is_live && $auction->ends_at_original->isPast()
            && $auction->ends_at_original->diffInMinutes()
               < 15) {
            $auction->makeSureAuctionTimerHasNotEnded();

            return;
        }

        Lot::endOverDueLots();

        if ($auction) {
            if ($auction->shouldEndAuction()) {
                $auction->endAuction();
            }
        } else {
            $auction->makeSureAuctionTimerHasNotEnded();
        }
    }

    /**
     * Add global stuff for Auction system
     */
    public static function footerDetails()
    {
        $twig = App::make('twig');
        $twig->display('Auction/footer_details.twig', [
            'has_pusher' => self::usePusherNotifications(),
            'pusher_key' => env('PUSHER_AUTH'),
            'pusher_cluster' => env('PUSHER_APP_CLUSTER'),
            'enabled_currencies' => Currency::getEnabledCurrencies(),
            'comparison_currencies' => Currency::getEnabledComparisonCurrencies(),
        ]);

    }
}