<?php

namespace Mtc\Auction;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
class Watchlist extends Model
{

    /**
     * Table name
     *
     * @var string $table
     */
    protected $table = 'auction_watchlist_lots';

    /**
     * Mass assignable attributes
     *
     * @var array $fillable
     */
    protected $fillable = [
        'lot_id',
        'member_id',
        'has_notification'
    ];

    /**
     * Relationship with Lot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function lot()
    {
        return $this->belongsTo(Lot::class, 'lot_id');
    }

    /**
     * Relationship with Member
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(AuctionMember::class, 'member_id');
    }

    /**
     * Check if user is watching lot
     *
     * @param $lot_id
     * @param Member $member
     * @return bool
     */
    public static function isUserWatching($lot_id, AuctionMember $member)
    {
        return self::query()
                   ->where('lot_id', $lot_id)
                   ->where('member_id', $member->id)
                   ->exists();
    }

    /**
     * Switch between user watching and not watching lot
     *
     * @param $lot_id
     * @param $already_watching
     * @param Member $member
     * @return Model
     */
    public static function toggleUserWatchingLot($lot_id, $already_watching, AuctionMember $member)
    {
        if (!$already_watching) {
            return self::query()
                       ->updateOrCreate([
                           'lot_id' => $lot_id,
                           'member_id' => $member->id
                       ]);
        }

        return self::query()
            ->where('lot_id', $lot_id)
            ->where('member_id', $member->id)
            ->delete();
    }



    /**
     * Send email about auction ending soon
     *
     * @param Auction $auction
     */
    public static function triggerEndingSoonNotifications(Auction $auction)
    {
        self::query()
            ->with('lot')
            ->whereIn('lot_id', $auction->lots->pluck('id'))
            ->get()
            ->groupBy('member_id')
            ->each(function ($lot_list, $member_id) {
                $twig = App::make('twig');
                $member = AuctionMember::query()->find($member_id);
                $subject = env('APP_NAME') . " - Lots in your watchlist end soon";
                $lot_list = $lot_list->map(function ($watchlist_lot) {
                    return $watchlist_lot->lot;
                });
                $mail = $twig->render('Auction/emails/ending_soon.twig', [
                    'lots' => $lot_list,
                    'member' => $member,
                ]);
                email($member->email, $subject , $mail);
            });

    }
}