<?php

namespace Mtc\AutoTraderStock\Imports;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Mtc\AutoTraderStock\Contracts\AutoTraderImportContract;
use Mtc\Core\Facades\Media;
use Mtc\Core\Images\ImageUploader;
use Mtc\Shop\Item;

class AutoTraderFileToShopItemImport extends AutoTraderImportContract implements ToCollection, WithHeadingRow
{
    private $image_uploader;

    public function __construct(ImageUploader $image_uploader)
    {
        $this->image_uploader = $image_uploader;
        $this->image_uploader->setFolders(Media::getFolders('product_folders'));
    }

    /**
     * @param Collection $collection
     */
    public function collection(Collection $collection)
    {
        $collection->each(function ($vehicle_data) {
            try {
                $this->updateSyncedData($vehicle_data);
            } catch (ModelNotFoundException $exception) {
                $this->createNew($vehicle_data);
            }
        });

        $this->markOldVehiclesDeleted($collection->pluck('stock_item_id'));
    }

    protected function updateSyncedData(Collection $vehicle_data)
    {
        $vehicle = Item::query()->where('epos_code', $vehicle_data['stock_item_id'])->firstOrFail();

        $vehicle->fill([
            'deleted' => 0,
            'price' => $vehicle_data['price'],
        ]);

        $vehicle->save();
    }

    protected function createNew(Collection $vehicle_data)
    {
        /** @var Item $vehicle */
        $vehicle = Item::query()->create($this->mapDataToColumns($vehicle_data));

        $this->importImages($vehicle, explode(',', $vehicle_data['images']));
    }

    protected function mapDataToColumns($vehicle_data): array
    {
        if (!empty($vehicle_data['date_of_registration'])) {
            $date_of_registration = Carbon::createFromFormat('d-m-Y', $vehicle_data['date_of_registration']);
        } else {
            $date_of_registration = null;
        }

        return [
            'epos_code' => $vehicle_data['stock_item_id'],
            'autotrader_did' => $vehicle_data['did'],
            'autotrader_created_date' => $vehicle_data['created_date'],
            'registration_number' => $vehicle_data['registration'],
            'registration_code' => $vehicle_data['reg_code'],
            'make' => $vehicle_data['make'],
            'model' => $vehicle_data['model'],
            'name' => $vehicle_data['model'],
            'derivative' => $vehicle_data['derivative'],
            'attention_grabber' => $vehicle_data['attention_grabber'],
            'engine_size' => $vehicle_data['engine_size'],
            'engine_size_unit' => $vehicle_data['engine_size_unit'],
            'fuel_type' => $vehicle_data['fuel_type'],
            'year' => $vehicle_data['manufactured_year'],
            'mileage' => $vehicle_data['mileage'],
            'mileage_unit' => $vehicle_data['mileage_unit'],
            'body_style' => $vehicle_data['bodytype'],
            'colour' => $vehicle_data['colour'],
            'transmission' => $vehicle_data['transmission'],
            'doors' => $vehicle_data['doors'],
            'previous_owners' => $vehicle_data['previous_owners'],
            'price' => $vehicle_data['price'],
            'sort_price' => $vehicle_data['price'],
            'price_exvat' => $vehicle_data['price'] / 1.2,
            'price_extra' => $vehicle_data['price_extra'],
            'wheel_base_type' => $vehicle_data['wheel_base_type'],
            'cab_type' => $vehicle_data['cab_type'],
            'four_wheel_drive' => $vehicle_data['four_wheel_drive'],
            'franchise_approved' => $vehicle_data['franchise_approved'] === 'Y' ? 1 : 0,
            'style' => $vehicle_data['style'],
            'sub_style' => $vehicle_data['sub_style'],
            'hours' => $vehicle_data['hours'],
            'number_of_berths' => $vehicle_data['number_of_berths'],
            'axle' => $vehicle_data['axle'],
            'dealer_reference' => $vehicle_data['dealer_reference'],
            'video_url' => $vehicle_data['video_url'],
            'date_of_registration' => $date_of_registration,
            'service_history' => $vehicle_data['service_history'],
            'key_information' => $vehicle_data['key_information'],
            'description' => $vehicle_data['key_information'],
            'other_vehicle_text' => $vehicle_data['other_vehicle_text'],
            'closer' => $vehicle_data['closer'],
            'feature' => $vehicle_data['feature'],
            'vatstatus' => $vehicle_data['vatstatus'],
            'autotrader_id' => $vehicle_data['id'],
            'mot_date' => $vehicle_data['mot_date'],
            'vin' => $vehicle_data['vin'],
            'vat_rate' => config('settings.VAT_RATE'),
        ];
    }

    protected function importImages(Item $vehicle, $image_list)
    {
        collect($image_list)
            ->each(function ($image_url, $index) use ($vehicle) {
                try {
                    $file_path = $this->image_uploader->uploadFromUrl($image_url);
                    if (!empty($file_path['error'])) {
                        return;
                    }
                    $image = $vehicle->images()
                        ->create([
                            'name' => $file_path['name'],
                            'default' => $index == 0 ? 1 : 0,
                        ]);
                } catch (\Exception $exception) {
                    Log::warning('Failed to import image', [
                        'vehicle_id' => $vehicle->id,
                        'image_url' => $image_url,
                        'error' => $exception,
                    ]);
                }
            });
    }

    protected function markOldVehiclesDeleted($vehicle_ids)
    {
        Item::query()
            ->where('deleted', 0)
            ->whereNotIn('epos_code', $vehicle_ids)
            ->update([
                'deleted' => 1,
            ]);
    }
}
