<?php

namespace Mtc\AutoTraderStock\Providers;

use GuzzleHttp\Client;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Support\ServiceProvider;
use Mtc\AutoTraderStock\AutoTraderFileEngine;
use Mtc\AutoTraderStock\Console\Commands\ImportStockCommand;
use Mtc\AutoTraderStock\Contracts\AutoTraderApiImportContract;
use Mtc\AutoTraderStock\Contracts\AutoTraderEngineContract;
use Mtc\AutoTraderStock\Contracts\AutoTraderImportContract;
use Mtc\AutoTraderStock\Contracts\AutoTraderServiceContract;
use Mtc\AutoTraderStock\Services\AutoTraderApi;

class AutoTraderProvider extends ServiceProvider
{
    public function register()
    {
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/auto_trader_stock.php', 'auto_trader_stock');
    }

    public function boot()
    {
        if ($this->app->runningUnitTests()) {
            return;
        }
        $this->app->bind(AutoTraderImportContract::class, config('auto_trader_stock.file_import_map'));
        $this->app->bind(AutoTraderApiImportContract::class, config('auto_trader_stock.api_import_map'));

        $this->app->bind(AutoTraderFileEngine::class, function () {
            return new AutoTraderFileEngine(
                config('auto_trader_stock'),
                $this->app->make(AutoTraderImportContract::class)
            );
        });

        $this->app->bind(AutoTraderApi::class, function () {
            $config = config('auto_trader_stock', []);
            $config['is_production'] = in_array(config('app.env'), ['demo', 'production']);
            return new AutoTraderApi(
                $config,
                $this->app->make(Client::class)
            );
        });

        $this->app->bind(AutoTraderServiceContract::class, function () {
            $class = config('auto_trader_stock.api_service_class');
            return new $class(
                config('auto_trader_stock.merchant_id'),
                $this->app->make(AutoTraderApi::class)
            );
        });

        $this->app->bind(AutoTraderEngineContract::class, config('auto_trader_stock.driver'));

        if (config('auto_trader_stock.enabled') === true) {
            $this->commands([
                ImportStockCommand::class,
            ]);

            $this->app->booted(function () {
                /** @var Schedule $schedule */
                $schedule = $this->app->make(Schedule::class);
                foreach (config('auto_trader_stock.sync_schedule', []) as $time_of_day) {
                    $schedule->command(ImportStockCommand::class)->dailyAt($time_of_day);
                }

                if (config('auto_trader_stock.perform_last_updated_sync')) {
                    $schedule->command(ImportStockCommand::class, [ '--last_updated' ])->everyFifteenMinutes();
                }
            });
        }
    }
}
