<?php

namespace Mtc\AutoTraderStock\Services;

use Carbon\Carbon;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class AutoTraderApi
{
    /**
     * @var Client
     */
    protected $client;

    /**
     * @var array
     */
    protected $config;

    public function __construct(array $config, Client $client)
    {
        $this->config = $config;
        $this->client = $client;
    }

    public function get(string $path)
    {
        $result = $this->client->get(
            $this->endpoint($path),
            [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken()
                ]
            ]
        );

        return json_decode((string)$result->getBody(), true);
    }

    public function accessToken(): ?string
    {
        return Cache::remember('auto-trader-api-access-token', Carbon::now()->addMinutes(15), function () {
            try {
                $payload = [
                    'key' => $this->config['api_key'],
                    'secret' => $this->config['api_secret']
                ];
                $result = $this->client->post(
                    $this->endpoint('authenticate'),
                    [
                        'form_params' => $payload
                    ]
                );

                return json_decode((string)$result->getBody())->access_token;
            } catch (ClientException $exception) {
                Log::error($exception->getMessage(), [
                    'response' => (string)$exception->getResponse()->getBody(),
                ]);
                return null;
            } catch (\Exception $exception) {
                Log::error($exception->getMessage());
                return null;
            }
        });
    }

    public function endpoint(string $path)
    {
        if ($this->config['is_production'] !== true || $this->config['use_sandbox']) {
            return 'https://api-sandbox.autotrader.co.uk/' . $path;
        }

        return 'https://api.autotrader.co.uk/' . $path;
    }
}
