<?php

namespace App\Modules\Stock;

use App\Contracts\StockProvider;
use App\Events\NewVehicleImported;
use App\Events\StockSyncFinished;
use App\Events\VehicleUpdatedFromImport;
use App\Facades\Settings;
use App\Jobs\ImportImagesFromUrlList;
use App\TaxonomyMap;
use App\Traits\ImportChecksConditions;
use App\Traits\MapsTaxonomies;
use App\Traits\StockSyncTraits;
use App\Traits\VehicleRegistrationDateFinderTrait;
use App\VehicleType;
use Carbon\Carbon;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Currency;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Services\FinanceService;
use Mtc\MercuryDataModels\Services\FinanceServiceHelper;
use Mtc\MercuryDataModels\Vehicle;

class Skupenet implements StockProvider
{
    use DispatchesJobs;
    use MapsTaxonomies;
    use StockSyncTraits;
    use VehicleRegistrationDateFinderTrait;
    use ImportChecksConditions;

    private Collection $dealerships;
    private array $make = [];
    private array $model = [];
    private array $transmission = [];
    private array $body_style = [];
    private array $fuel_type = [];

    private const ENDPOINT = 'https://feeds.dealerhub.ie/api/v1/stock?';
    /**
     * Check if enabled
     *
     * @return bool
     */
    public function enabled(): bool
    {
        return Settings::get('stock-skupenet-enabled', false) ?? false;
    }

    /**
     * Name of the integration
     *
     * @return string
     */
    public function name(): string
    {
        return 'Skupenet';
    }

    /**
     * Perform a scheduled import task
     *
     * @return void
     */
    public function runScheduledImport(bool $fullSync = true): void
    {
        Vehicle::query()
            ->where('stock_provider', $this->getProviderName())
            ->update([
                'was_recently_synced' => false,
            ]);
        $this->import($this->fetchVehicles());
    }

    /**
     * Fields to add to dealership management
     *
     * @return array[]
     */
    public function dealershipAdditionalDataFields()
    {
        return [
            'skupenet-location-id' => [
                'type' => 'text',
                'label' => 'Seller ID within Skupenet DMS',
            ],
        ];
    }

    protected function getProviderName(): string
    {
        return 'skupenet';
    }

    protected function getDetailsForTaxonomyMap(array $record): array
    {
        return [
            'tenant_id' => tenant('id'),
            'registration_number' => $record['attributes']['registration'] ?? null,
            'make' => $record['attributes']['make'] ?? '',
            'model' => $record['attributes']['model'] ?? '',
        ];
    }

    /**
     * Fetch vehicles from API
     *
     * @return Collection
     */
    private function fetchVehicles(): Collection
    {
        return $this->getStockForLocations($this->getAllStockLocations('skupenet-location-id'))
            ->unique('id');
    }

    private function getStockForLocations(Collection $locations): Collection
    {
        $params = [
            'dealerIds' => $locations->implode(','),
            'pageSize' => 50,
            'page' => 1
        ];
        $response = Http::withHeaders(['X-API-KEY' => config('services.skupenet.api_key')])
            ->get(self::ENDPOINT . http_build_query($params));

        if (!$response->successful()) {
            throw new \Exception('Failed to retrieve vehicle stock from Skupenet ' . tenant('id'));
        }
        $vehicles = collect($response->json('stock'));
        $numPages = $response->json('paging.totalPages') ?? 1;

        for ($page = 2; $page < $numPages; $page++) {
            $params['page'] = $page;
            $response = Http::withHeaders(['X-API-KEY' => config('services.skupenet.api_key')])
                ->get(self::ENDPOINT . http_build_query($params));

            $vehicles = $vehicles->merge($response->json('stock'));
        }

        return $vehicles;
    }

    /**
     * Import vehicles
     *
     * @param Collection $stock
     * @return void
     */
    private function import(Collection $stock)
    {
        if ($stock->count() === 0) {
            // No vehicles to sync, assume service failure and stop sync to avoid removing vehicles
            return;
        }
        $stock = $stock->filter(fn($vehicle) => $this->filterByPrice($vehicle));

        $stock->each(fn(array $vehicle) => $this->syncVehicle($vehicle));

        $this->removeOld($stock);
        Event::dispatch(new StockSyncFinished($this->getProviderName()));
    }

    /**
     * Filter out vehicles depending on price settings
     *
     * @param array $vehicle
     * @return bool
     */
    private function filterByPrice(array $vehicle): bool
    {
        if (Settings::get('stock-skupenet-sync-with-price') !== true) {
            return true;
        }

        if (empty($vehicle['price']) || !is_numeric($vehicle['price'])) {
            return false;
        }

        return $vehicle['price'] > 0;
    }

    /**
     * Sync vehicle record
     *
     * @param array $data
     * @return void
     */
    private function syncVehicle(array $data): void
    {
        $type = match (true) {
            stripos($data['stockType'], 'lcv') => VehicleType::LCV->value,
            stripos($data['stockType'], 'bike'), stripos($data['stockType'], 'rad') => VehicleType::MOTORCYCLE->value,
            default => VehicleType::CAR->value,
        };
        if ($data['attributes']['bodyType'] === 'Commercial/Van') {
            $type = VehicleType::LCV->value;
        }

        $make_id = $this->getMappedTaxonomy(TaxonomyMap::MAKE, $data['attributes']['make'] ?? '', $data);

        /** @var Vehicle $vehicle */
        /** @var Vehicle $vehicle */
        $vehicle = Vehicle::query()
            ->where('uuid', $data['id'])
            ->firstOrNew([
                'stock_provider' => $this->getProviderName(),
                'uuid' => $data['id'],
            ]);
        $mileageType = strtolower($data['attributes']['mileageUnit'] ?? 'kms') === 'km' ? 'km' : 'mi';

        if ($this->shouldSyncVideo($vehicle, $data)) {
            $vehicle['main_video_url'] = $this->getVideoUrl($data);
        }

        if (!empty($data['attributes']['make']) && !empty($data['attributes']['model'])) {
            $title = $data['attributes']['make'] . ' ' . $data['attributes']['model'];
        } else {
            $title = $data['title'];
        }

        $description_data = explode("\n\n", $data['description'] ?? '');
        if (count($description_data) > 1) {
            // We have no clear way of knowing if this is description or feature list since clients can
            // use double newline as well in description, but we assume that over 5 newlines will be features
            if (count(explode("\n", end($description_data))) > 5) {
                $features = array_pop($description_data);
            }
            $description_data[0] = implode("\n", $description_data);
        }
        $vehicle->fill([
            'title' => $title,
            'registration_number' => $data['attributes']['registration'] ?? null,
            'first_registration_date' => $this->determineRegistrationDate(),
            'derivative' => $data['attributes']['trim'] ?? '',
            'make_id' => $make_id,
            'model_id' => $this->getMappedTaxonomy(
                TaxonomyMap::MODEL,
                $data['attributes']['model'] ?? '',
                $data,
                $make_id
            ),
            'colour' => $data['attributes']['colour'] ?? null,
            'fuel_type_id' => $this->getMappedTaxonomy(
                TaxonomyMap::FUEL_TYPE,
                $data['attributes']['fuelType'] ?? null,
                $data
            ),
            'body_style_id' => $this->getMappedTaxonomy(
                TaxonomyMap::BODY_STYLE,
                $data['attributes']['bodyType'] ?? null,
                $data
            ),
            'type' => $type,
            'bhp' => $data['attributes']['enginePowerBhp'] ?? null,
            'engine_size_cc' => $data['attributes']['engineSizeCC'] ?? null,
            'seats' => $data['attributes']['numSeats'] ?? null,
            'previous_owner_count' => $data['attributes']['numOwners'] ?? null,
            'door_count' => $data['attributes']['numDoors'] ?? null,
            'transmission_id' => $this->getMappedTaxonomy(
                TaxonomyMap::TRANSMISSION,
                $data['attributes']['transmission'] ?? null,
                $data
            ),
            'manufacture_year' => $data['attributes']['year'] ?? null,
            'was_recently_synced' => true,
            'is_new' => ($data['attributes']['mileage'] ?? 1)
                <= Settings::get('stock-skupenet-new-vehicle-mileage-threshold'),
            'dealership_id' => $this->dealershipId($data['dhfDealerId']),
            'odometer_mi' => $mileageType === 'km'
                ? $vehicle->kmToMiles($data['attributes']['mileage'] ?? 0)
                : $data['attributes']['mileage'] ?? 0,
            'odometer_km' => $mileageType === 'km'
                ? $data['attributes']['mileage'] ?? 0
                : $vehicle->milesToKm($data['attributes']['mileage'] ?? 0),
            'price' => $this->getPrice($data),
            'description' => trim(strip_tags($description_data[0])),
        ]);
        $vehicle->is_published = $this->shouldBePublished(fn() => $data['price'] > 1, 'skupenet', $vehicle);
        $vehicle->save();

        if ($this->shouldSyncImages($vehicle, $data)) {
            $this->syncImages($vehicle, $data['images'] ?? []);
        }

        if (!empty($features)) {
            $this->syncOptions($vehicle, explode("\n", $features));
        }

        $this->storeUnmappedTaxonomy($vehicle);

        if ($vehicle->wasRecentlyCreated) {
            Event::dispatch(new NewVehicleImported($vehicle, $data, $this->getProviderName()));
        } else {
            Event::dispatch(new VehicleUpdatedFromImport($vehicle, $data, $this->getProviderName()));
        }
    }

    /**
     * Remove vehicles that are not part of the stock feed
     *
     * @param Collection $vehicles
     * @return void
     */
    private function removeOld(Collection $vehicles): void
    {
        Vehicle::query()
            ->where('stock_provider', $this->getProviderName())
            ->where('was_recently_synced', 0)
            ->delete();
    }

    /**
     * Check if images should be synced for vehicles
     *
     * @param Vehicle $vehicle
     * @param array $data
     * @return bool
     */
    private function shouldSyncImages(Vehicle $vehicle, array $data): bool
    {
        $images = collect($data['images'] ?? [])->map(fn($image) => rtrim($image['url'], '?'))->toArray();
        $images_from_other_provider = $vehicle->mediaUses()
            ->whereHas('media', fn($query) => $query->where('image_provider', '!=', 'skupenet'))
            ->count();
        return Settings::get('stock-skupenet-sync-images')
            && $images_from_other_provider == 0
            && $this->imageChecksumMismatch($images, $vehicle);
    }

    private function shouldSyncVideo(Vehicle $vehicle, array $data): bool
    {
        return Settings::get('stock-skupenet-sync-video', false);
    }

    /**
     * Sync images for vehicle
     *
     * @param Vehicle $vehicle
     * @param array $images
     * @return void
     */
    private function syncImages(Vehicle $vehicle, array $images)
    {
        $imageList = collect($images)
         ->sortBy('sortOrder')
        ->map(fn($image) => rtrim($image['url'], '?'))
         ->values();

        $to_remove = $this->getImagesToRemove($vehicle, $imageList->toArray());
        MediaUse::query()->whereIn('id', $to_remove)->delete();
        $this->dispatch(new ImportImagesFromUrlList($imageList, $vehicle, false, 'skupenet'));
    }

    /**
     * handle engine sizes cc which may contain text e.g. 'litre'
     * @param bool $value
     */
    private function getEngineSize($value = false)
    {
        $value = strtoupper($value);
        $value = str_replace('LITRES', '', $value);
        $value = str_replace('LITRE', '', $value);
        $value = str_replace(' ', '', $value);

        if (is_numeric($value)) {
            return $value * 1000;
        }

        return false;
    }

    /**
     * Get currency type from value
     *
     * @param string $value
     * @return string
     */
    private function getCurrencyType(string $value = ''): string
    {
        $type = 'EUR';

        if (stripos($value, 'gbp') !== false || stripos($value, 'pound') !== false) {
            $type = 'GBP';
        }

        return $type;
    }

    /**
     * Get price in default currency
     *
     * @param array $data
     * @return float|int|string
     */
    private function getPrice(array $data = []): float|int|string
    {
        $price = is_numeric($data['price']) ? $data['price'] : 0;
        $priceCurrency = $this->getCurrencyType($data['currency']);

        if ($priceCurrency == 'EUR') {
            $currency = Currency::query()
                ->where('CODE', Settings::get('app-details-currency'))
                ->first();

            return $currency->ratio * $price;
        }

        if ($priceCurrency != Settings::get('app-details-currency')) {
            $priceCurrencyRate = Currency::query()
                ->where('CODE', $priceCurrency)
                ->first();

            $priceInEuro = $priceCurrencyRate->ratio * $price;

            $appCurrency = Currency::query()
                ->where('CODE', Settings::get('app-details-currency'))
                ->first();

            return $appCurrency->ratio * $priceInEuro;
        }

        return $price;
    }

    private function shouldSyncOptions(Vehicle $vehicle, array $options): bool
    {
        return !empty($options)
            && $vehicle->features()->count() === 0;
    }

    private function syncOptions(Vehicle $vehicle, array $options)
    {
        collect($options)
            ->each(fn($optionData) => $vehicle->features()->updateOrCreate([
                'vehicle_type' => $vehicle->getMorphClass(),
                'name' => trim(str_replace("•", "", $optionData)),
            ]));
    }

    private function getVideoUrl(array $data): ?string
    {
        if (empty($data['videos'][0]['url'])) {
            return null;
        }

        try {
            return $data['videos'][0]['url'];
        } catch (\Exception $exception) {
            return null;
        }
    }

    private function getImagesToRemove(?Vehicle $vehicle, array $images)
    {
        $media_uses = $vehicle->mediaUses()->with('media')
            ->whereHas('media', fn($query) => $query->where('image_provider', 'skupenet'))
            ->get();

        $not_in_list = $media_uses
            ->filter(fn($mediaUse) => !in_array($mediaUse->media->source_filename, $images))
            ->pluck('id');

        $duplicates = $media_uses->groupBy(fn($mediaUse) => $mediaUse->media->source_filename)
            ->filter(fn(Collection $group) => $group->count() > 1)
            ->map(fn(Collection $group) => $group->first())
            ->pluck('id');

        return $not_in_list->merge($duplicates);
    }

    private function determineRegistrationDate(): ?string
    {
        return !empty($data['attributes']['registration'])
            ? Carbon::parse($this->getRegistrationDateFromRegistrationNumber($data['attributes']['registration']))
            : null;
    }
}
