<?php

namespace App\Console\Commands;

use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;
use Illuminate\Console\Command;

class SetKeyloopMakesAndModels extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'vehicles:set-keyloop-makes-and-models';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Set the Keyloop IDs for vehicle makes and models';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->setKeyloopMakeIds();
        $this->setKeyloopModelIds();
    }

    /**
     * @return void
     */
    private function setKeyloopMakeIds(): void
    {
        collect($this->makeMappings())->each(function ($make_name, $keyloop_id) {
            VehicleMake::query()
                ->where('name', '=', $make_name)
                ->whereNull('keyloop_id')
                ->update([
                    'keyloop_id' => $keyloop_id
                ]);
        });
    }

    /**
     * @return void
     */
    private function setKeyloopModelIds(): void
    {
        VehicleModel::query()
            ->whereNull('keyloop_id')
            ->get()
            ->each(function (VehicleModel $model) {
                $model->keyloop_id = strtoupper($model->name);

                // handle custom Keyloop formatting
                $model->keyloop_id = match ($model->keyloop_id) {
                    'E-TRON SPORTBACK' => 'E-TRON SPORTBAC',
                    '1 SERIES' => 'SERIES 1',
                    '2 SERIES' => 'SERIES 2',
                    '2 SERIES ACTIVE TOURER' => 'SERIES 2 ACTIVE',
                    '3 SERIES' => 'SERIES 3',
                    '4 SERIES' => 'SERIES 4',
                    '5 SERIES' => 'SERIES 5',
                    '6 SERIES' => 'SERIES 6',
                    '7 SERIES' => 'SERIES 7',
                    '8 SERIES' => 'SERIES 8',
                    'GRAND C4 SPACETOURER' => 'GRAND C4 SPACET',
                    'RANGE ROVER EVOQUE' => 'RANGE ROVER EVO',
                    'RANGE ROVER SPORT' => 'RANGE ROVER SPO',
                    'RANGE ROVER VELAR' => 'RANGE ROVER VEL',
                    'A CLASS' => 'A-CLASS',
                    'B CLASS' => 'B-CLASS',
                    'C CLASS' => 'C-CLASS',
                    'E CLASS' => 'E-CLASS',
                    'GLC CLASS' => 'GLC-CLASS',
                    'S CLASS' => 'S-CLASS',
                    'V CLASS' => 'V-CLASS',
                    default => $model->keyloop_id,
                };

                $model->save();
            });
    }

    /**
     * @return string[]
     */
    private function makeMappings(): array
    {
        return [
            'ALF' => 'Alfa Romeo',
            'AUD' => 'Audi',
            'BMW' => 'BMW',
            'CIT' => 'Citroen',
            'CUP' => 'Cupra',
            'DAC' => 'Dacia',
            'DS' => 'DS',
            'FIA' => 'Fiat',
            'FOR' => 'Ford',
            'HON' => 'Honda',
            'HYU' => 'Hyundai',
            'JAG' => 'Jaguar',
            'JEE' => 'Jeep',
            'KIA' => 'KIA',
            'LAN' => 'Land Rover',
            'LEX' => 'Lexus',
            'MAN' => 'MAN',
            'MAX' => 'Maxus',
            'MAZ' => 'Mazda',
            'MC' => 'BMW Motorrad',
            'MER' => 'Mercedes-Benz',
            'MG' => 'MG',
            'MIN' => 'MINI',
            'MIT' => 'Mitsubishi',
            'NIS' => 'Nissan',
            'OPE' => 'Opel',
            'PEU' => 'Peugeot',
            'POR' => 'Porsche',
            'REN' => 'Renault',
            'SEA' => 'SEAT',
            'SKO' => 'Skoda',
            'SUB' => 'Subaru',
            'SUZ' => 'Suzuki',
            'TES' => 'Tesla',
            'TOY' => 'Toyota',
            'VAG' => 'Volkswagen',
            'VOL' => 'Volvo',
        ];
    }
}
