<?php

namespace Tests\Feature\Controllers;

use App\Notifications\PasswordResetNotification;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Notification;
use Mtc\MercuryDataModels\User;
use Tests\TestCase;
use Tests\UserForTenant;

class AllUserControllerTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        User::factory(5)->create();

        $response = $this->actingAs($this->getUser())
            ->getJson(route('all-users.index', [], false));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertCount(6, $response->json('data')); // +1 from the initial seed (acting user)
        $this->assertArrayHasKey('email', $response->json('data.0'));
        $this->assertArrayHasKey('name', $response->json('data.0'));
    }

    public function testShow()
    {
        $user = User::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->putJson(route('user-management.reset-two-factor', $user));

        $response->assertStatus(200);
        $this->assertEquals($user->id, $response->json('id'));
        $this->assertArrayHasKey('name', $response->json());
        $this->assertArrayHasKey('email', $response->json());
        $this->assertArrayHasKey('requires_password_change', $response->json());
        $this->assertArrayHasKey('two_factor_enabled', $response->json());
    }

    public function testDestroy()
    {
        $user = User::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->deleteJson(route('all-users.destroy', $user));

        $response->assertStatus(200);
        $this->assertNull(User::query()->find($user->id));
    }

    public function testReset2FA()
    {
        $user = User::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->putJson(route('user-management.reset-two-factor', $user));

        $response->assertStatus(200);
        $this->assertEquals($user->id, $response->json('id'));
        $this->assertFalse($response->json('two_factor_enabled'));
    }

    public function testRequirePasswordChange()
    {
        $user = User::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->putJson(route('user-management.require-password-change', $user));

        $response->assertStatus(200);
        $this->assertEquals($user->id, $response->json('id'));
        $this->assertTrue($response->json('requires_password_change'));
    }

    public function testResetPassword()
    {
        Notification::fake();
        $user = User::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->putJson(route('user-management.reset-password', $user));

        $response->assertStatus(200);
        $this->assertEquals($user->id, $response->json('id'));
        $this->assertTrue($response->json('requires_password_change'));
        Notification::assertSentTo($user, PasswordResetNotification::class);
    }
}
