<?php

namespace Tests\Feature\Controllers;

use App\Models\ReleaseNote;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Tests\UserForTenant;

class ReleaseNoteControllerTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

     /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        ReleaseNote::factory(3)->create();

        $response = $this->actingAs($this->getUser())
            ->getJson(route('release-notes.index'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertCount(3, $response->json('data'));
    }

    public function testStore()
    {
        $response = $this->actingAs($this->getUser())
            ->postJson(route('release-notes.store'), [
                'title' => 'foo'
            ]);

        $response->assertStatus(201);
        $this->assertEquals('foo', $response->json('title'));
        $this->assertTrue(ReleaseNote::query()->where('title', 'foo')->exists());

    }

    public function testUpdate()
    {
        $releaseNote = ReleaseNote::factory()->create(['title' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->putJson(route('release-notes.update', $releaseNote->id), [
                'is_published' => true,
                'published_at' => '2025-05-01',
                'title' => 'baz',
                'content' => 'lorem ipsum dolor',
            ]);

        $releaseNote->refresh();
        $response->assertStatus(200);
        $this->assertEquals('baz', $releaseNote->title);
        $this->assertEquals('lorem ipsum dolor', $releaseNote->content);
        $this->assertEquals('2025-05-01', $releaseNote->published_at->format('Y-m-d'));
        $this->assertTrue($releaseNote->is_published);

    }

    public function testShow()
    {
        $releaseNote = ReleaseNote::factory()->create(['title' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->getJson(route('release-notes.show', $releaseNote->id));

        $response->assertStatus(200);
        $this->assertEquals($releaseNote->id, $response->json('id'));
        $this->assertEquals('foo', $response->json('title'));

    }

    public function testDelete()
    {
        $releaseNote = ReleaseNote::factory()->create(['title' => 'foo']);
        $response = $this->actingAs($this->getUser())
            ->deleteJson(route('release-notes.destroy', $releaseNote->id));

        $response->assertStatus(200);
        $this->assertFalse(ReleaseNote::query()->where('id', $releaseNote->id)->exists());
    }
}
