<?php

namespace Tests\Unit;

use App\Models\ExportMap;
use App\ExportMapRepository;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Tests\TestCase;
use Carbon\Carbon;

class ExportMapRepositoryTest extends TestCase
{
    use RefreshDatabase;

    protected $tenancy = true;

    public function test_column_count_and_header_mapping_is_correct()
    {
        $vehicle = Vehicle::factory()->create([
            'registration_number' => 'TEST01',
            'vin' => 'VIN12345',
        ]);

        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['registration_number', 'vin'],
            'headers' => ['Reg Number', 'VIN Number'],
        ]);

        // 1. Test column count
        $this->assertCount(2, $mappedData);

        // 2. Test that columns match headers in order
        $this->assertEquals($vehicle->registration_number, $mappedData[0]);
        $this->assertEquals($vehicle->vin, $mappedData[1]);
    }

    /**
     * Helper method to run the mapping process for a single entry.
     * It uses Reflection to test the protected 'mapEntries' method directly.
     *
     * @param Vehicle $vehicle The data entry to process.
     * @param array $mapConfig The configuration for the ExportMap.
     * @return array The resulting mapped data row.
     */
    private function runMapProcessing(Vehicle $vehicle, array $mapConfig): array
    {
        // Ensure default values are set if not provided
        $config = array_merge([
            'name' => 'Test Map',
            'type' => 'vehicle',
            'columns' => [],
            'mapping_rules' => [],
            'headers' => [],
        ], $mapConfig);

        $exportMap = new ExportMap($config);
        $repository = new ExportMapRepository($exportMap);

        $method = new \ReflectionMethod(ExportMapRepository::class, 'mapEntries');
        $method->setAccessible(true);

        $resultCollection = $method->invoke($repository, collect([$vehicle]));

        return $resultCollection->first() ?? [];
    }

    // --- Operator Tests ---

    public function test_operator_equals()
    {
        $vehicle = Vehicle::factory()->create(['type' => 'Car']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'type', 'op' => 'eq', 'value' => 'Car'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Is a Car']],
            ]],
        ]);
        $this->assertEquals('Is a Car', $mappedData[0]);

        $vehicle->type = 'LCV';
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'type', 'op' => 'eq', 'value' => 'Car'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Is a Car']],
            ]],
        ]);
        $this->assertNull($mappedData[0]);
    }

    public function test_operator_not_equals()
    {
        $vehicle = Vehicle::factory()->create(['type' => 'LCV']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'type', 'op' => 'neq', 'value' => 'Car'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Not a Car']],
            ]],
        ]);
        $this->assertEquals('Not a Car', $mappedData[0]);
    }

    public function test_operator_is_any_of()
    {
        $vehicle = Vehicle::factory()->create(['colour' => 'Blue']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'colour', 'op' => 'in', 'value' => 'Red,Green,Blue'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Primary Color']],
            ]],
        ]);
        $this->assertEquals('Primary Color', $mappedData[0]);
    }

    public function test_operator_is_none_of()
    {
        $vehicle = Vehicle::factory()->create(['colour' => 'Black']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'colour', 'op' => 'not_in', 'value' => 'Red,Green,Blue'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Not a Primary Color']],
            ]],
        ]);
        $this->assertEquals('Not a Primary Color', $mappedData[0]);
    }

    public function test_operator_contains()
    {
        $vehicle = Vehicle::factory()->create(['description' => 'A beautiful family car']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'description', 'op' => 'contains', 'value' => 'family'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Contains Keyword']],
            ]],
        ]);
        $this->assertEquals('Contains Keyword', $mappedData[0]);
    }

    public function test_operator_starts_with()
    {
        $vehicle = Vehicle::factory()->create(['vin' => 'VIN12345']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'vin', 'op' => 'starts_with', 'value' => 'VIN'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Prefix Match']],
            ]],
        ]);
        $this->assertEquals('Prefix Match', $mappedData[0]);
    }

    public function test_operator_ends_with()
    {
        $vehicle = Vehicle::factory()->create(['vin' => 'VIN12345XYZ']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'vin', 'op' => 'ends_with', 'value' => 'XYZ'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Suffix Match']],
            ]],
        ]);
        $this->assertEquals('Suffix Match', $mappedData[0]);
    }

    public function test_operator_greater_than()
    {
        $vehicle = Vehicle::factory()->create(['price' => 20000]);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'price', 'op' => 'gt', 'value' => '15000'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Expensive']],
            ]],
        ]);
        $this->assertEquals('Expensive', $mappedData[0]);
    }

    public function test_operator_greater_or_equal()
    {
        $vehicle = Vehicle::factory()->create(['manufacture_year' => 2022]);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'manufacture_year', 'op' => 'gte', 'value' => '2022'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Modern Car']],
            ]],
        ]);
        $this->assertEquals('Modern Car', $mappedData[0]);
    }

    public function test_operator_less_than()
    {
        $vehicle = Vehicle::factory()->create(['odometer_mi' => 50000]);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'odometer_mi', 'op' => 'lt', 'value' => '60000'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Low Mileage']],
            ]],
        ]);
        $this->assertEquals('Low Mileage', $mappedData[0]);
    }

    public function test_operator_less_or_equal()
    {
        $vehicle = Vehicle::factory()->create(['previous_owner_count' => 1]);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'previous_owner_count', 'op' => 'lte', 'value' => '1'],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Few Owners']],
            ]],
        ]);
        $this->assertEquals('Few Owners', $mappedData[0]);
    }

    public function test_operator_is_empty()
    {
        $vehicle = Vehicle::factory()->create(['trim' => null]);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'trim', 'op' => 'is_empty', 'value' => null],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Trim Not Specified']],
            ]],
        ]);
        $this->assertEquals('Trim Not Specified', $mappedData[0]);
    }

    public function test_operator_is_not_empty()
    {
        $vehicle = Vehicle::factory()->create(['trim' => 'Sportline']);
        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['custom1'],
            'mapping_rules' => [[
                'when' => ['field' => 'trim', 'op' => 'is_not_empty', 'value' => null],
                'then' => [['action' => 'set', 'target' => 'custom1', 'value' => 'Trim Is Specified']],
            ]],
        ]);
        $this->assertEquals('Trim Is Specified', $mappedData[0]);
    }

    // --- Other Tests ---

    public function test_value_casting_for_export()
    {
        $date = Carbon::create(2025, 5, 20);
        $vehicle = Vehicle::factory()->create([
            'is_published' => true,
            'is_sold' => false,
            'first_registration_date' => $date,
        ]);

        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['is_published', 'is_sold', 'first_registration_date'],
        ]);

        $this->assertEquals('Y', $mappedData[0]);
        $this->assertEquals('N', $mappedData[1]);
        $this->assertEquals('2025-05-20', $mappedData[2]);
    }

    public function test_relationship_and_custom_attribute_mapping()
    {
        $make = VehicleMake::factory()->create(['name' => 'Volkswagen']);
        $model = VehicleModel::factory()->create(['name' => 'Golf']);
        $vehicle = Vehicle::factory()->create(['make_id' => $make->id, 'model_id' => $model->id]);

        $mappedData = $this->runMapProcessing($vehicle, [
            'columns' => ['make.name', 'model.name', 'empty'],
        ]);

        $this->assertEquals('Volkswagen', $mappedData[0]);
        $this->assertEquals('Golf', $mappedData[1]);
        $this->assertEquals('', $mappedData[2]);
    }
}
