<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Mtc\MercuryDataModels\Invoice;
use Mtc\MercuryDataModels\InvoiceStatus;
use Tests\TestCase;
use Tests\UserForTenant;

class MasterInvoiceControllerTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        Invoice::factory(5)->create();
        $response = $this->actingAs($this->getUser())
            ->getJson(route('invoices.index'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertEquals(5, $response->json('total'));
    }

    public function testShow()
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => 5,
            'email' => 'jdoe@example.com',
            'status' => InvoiceStatus::OPEN->value,
            'amount' => 123,
            'amount_paid' => 0,
            'vat_rate' => Invoice::REGULAR_VAT,
        ]);
        $response = $this->actingAs($this->getUser())
            ->getJson(route('invoices.show', $invoice));

        $response->assertStatus(200);
        $this->assertEquals($invoice->id, $response->json('id'));
        $this->assertEquals(5, $response->json('tenant_id'));
        $this->assertEquals(123, $response->json('amount'));
        $this->assertEquals(0, $response->json('amount_paid'));
        $this->assertEquals(InvoiceStatus::OPEN->value, $response->json('status'));
        $this->assertEquals(Invoice::REGULAR_VAT, $response->json('vat_rate'));
        $this->assertEquals('jdoe@example.com', $response->json('email'));
        $this->assertArrayHasKey('history', $response->json());
        $this->assertArrayHasKey('items', $response->json());
        $this->assertArrayHasKey('addresses', $response->json());
        $this->assertArrayHasKey('payments', $response->json());
    }

    public function testIndexWithStatusFilter()
    {
        Invoice::factory(5)->create(['status' => InvoiceStatus::OPEN->value]);
        Invoice::factory(3)->create(['status' => InvoiceStatus::CANCELLED->value]);

        $response = $this->actingAs($this->getUser())
            ->getJson(route('invoices.index', [
                'status' => InvoiceStatus::OPEN->value,
            ]));
        $response->assertStatus(200);
        $this->assertEquals(5, $response->json('total'));

        $response = $this->actingAs($this->getUser())
            ->getJson(route('invoices.index', [
                'status' => InvoiceStatus::CANCELLED->value,
            ]));
        $response->assertStatus(200);
        $this->assertEquals(3, $response->json('total'));
    }

    public function testIndexWithTenantFilter()
    {
        Invoice::factory(5)->create(['tenant_id' => 'abc123']);
        Invoice::factory(4)->create(['tenant_id' => 'dev123123']);

        $response = $this->actingAs($this->getUser())
            ->getJson(route('invoices.index', [
                'tenant_id' => 'abc123',
            ]));
        $response->assertStatus(200);
        $this->assertEquals(5, $response->json('total'));

        $response = $this->actingAs($this->getUser())
            ->getJson(route('invoices.index', [
                'tenant_id' => 'dev123123',
            ]));
        $response->assertStatus(200);
        $this->assertEquals(4, $response->json('total'));
    }
}
