<?php

namespace App\Imports\Traits;

use App\Facades\Settings;
use App\Jobs\ImportImagesFromUrlList;
use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;
use App\Models\ImportMap;
use App\Models\ImportMapField;
use App\TaxonomyMap;
use App\VehicleType;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Str;
use Mtc\ContentManager\Facades\Media;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Country;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\DrivetrainType;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\Vehicle;

trait VehicleImportFlow
{
    protected static array $allowedFieldsToUpdate = [];

    public function setAdditionalTasks(array $tasks): self
    {
        $this->additional_tasks = $tasks;
        return $this;
    }

    public function setImportMap(ImportMap $importMap): self
    {
        $this->import_map = $importMap;
        return $this;
    }

    private function processRow($row): ?Vehicle
    {
        if (empty($row)) {
            return null;
        }

        $search_params = $this->importRowSearchParams($row);

        if ($search_params === null) {
            return null;
        }

        /** @var Vehicle $model */
        $model = Vehicle::query()->firstOrNew($search_params);

        $model->fill($this->getFillableData($row, $model));
        if ($model->isDirty()) {
            $model->save();
        }

        $this->syncImages($model, $row['images'] ?? '');
        $this->syncFeatures($model, $row['features'] ?? '');

        return $model;
    }

    private function importRowSearchParams($row): ?array
    {
        if (!empty($row['slug'])) {
            $search_params = [
                'slug' => $row['slug'],
            ];
        } elseif (empty($row['uuid'])) {
            if (empty($row['registration_number'])) {
                return null;
            }
            $row['uuid'] = 'import-' . $row['registration_number'];
        }

        if (empty($search_params)) {
            $search_params = [
                'uuid' => $row['uuid'],
            ];
            if (!empty($row['stock_provider'])) {
                $search_params['stock_provider'] = $row['stock_provider'];
            }
        }

        return $search_params;
    }

    private function syncImages(Vehicle $vehicle, string $images): void
    {
        $imageList = collect(explode('|', $images))
            ->filter()
            ->reject(fn(string $image) => $this->imageAlreadyHostedOnServer($image));

        if ($imageList->isNotEmpty()) {
            $this->dispatch(new ImportImagesFromUrlList($imageList, $vehicle));
        }
    }

    private function syncFeatures(Vehicle $vehicle, string $features): void
    {
        collect(explode('|', $features))
            ->filter()
            ->each(fn($feature) => $vehicle->features()->updateOrCreate(['name' => $feature]));
    }

    private function imageAlreadyHostedOnServer($image): bool
    {
        try {
            return Media::assetExists($image);
        } catch (Exception $exception) {
            return false;
        }
    }

    private function getFillableData(array $vehicle, $model): array
    {
        if (empty(self::$vehicleFillable)) {
            self::$vehicleFillable = (new Vehicle())->getFillable();
        }

        if ($model->exists) {
            $tenantId = tenant('id');

            if (empty(self::$allowedFieldsToUpdate[$tenantId])) {
                self::$allowedFieldsToUpdate[$tenantId] = $this->import_map->fields
                    ->filter(fn(ImportMapField $field) => $field->data['save_on_update'] ?? true)
                    ->pluck('on_model')
                    ->filter()
                    ->toArray();
            }

            $vehicle =  array_intersect_key($vehicle, array_flip(self::$allowedFieldsToUpdate[$tenantId]));
        }

        $vehicle['was_recently_synced'] = true;
        if (empty($vehicle['stock_provider'])) {
            $vehicle['stock_provider'] = $this->getProviderName();
        }

        return collect($vehicle)
            ->mapWithKeys(fn($value, $field) => [
                $this->getMappedField($field) => $this->getMappedValue($field, $value, $vehicle)
            ])
            ->only(self::$vehicleFillable)
            ->toArray();
    }

    private function getMappedField($field): string
    {
        return match ($field) {
            'make' => 'make_id',
            'model' => 'model_id',
            'dealership' => 'dealership_id',
            'fuel_type' => 'fuel_type_id',
            'body_style' => 'body_style_id',
            'drivetrain' => 'drivetrain_id',
            'transmission' => 'transmission_id',
            default => $field,
        };
    }

    private function getMappedValue($field, $value, array $vehicle)
    {
        return match ($field) {
            'make' => $value && !empty($this->makes[$value])
                ? $this->makes[$value]
                : $this->getMappedTaxonomy(
                    TaxonomyMap::MAKE,
                    $value,
                    $vehicle
                ),
            'model' => !empty($value)
                ? $this->getMappedTaxonomy(
                    TaxonomyMap::MODEL,
                    $value,
                    $vehicle,
                    $this->getMappedTaxonomy(
                        TaxonomyMap::MAKE,
                        $vehicle['make'],
                        $vehicle
                    )
                ) : null,
            'dealership' => $value && !empty($this->dealerships[$value])
                ? $this->dealerships[$value]
                : $this->dealershipId($value),
            'fuel_type' => $value && !empty($this->fuel_types[$value])
                ? $this->fuel_types[$value]
                : $this->getMappedTaxonomy(
                    TaxonomyMap::FUEL_TYPE,
                    $value,
                    $vehicle
                ),
            'body_style' => $value && !empty($this->body_styles[$value])
                ? $this->body_styles[$value]
                : $this->getMappedTaxonomy(
                    TaxonomyMap::BODY_STYLE,
                    $value,
                    $vehicle
                ),
            'drivetrain' => $value && !empty($this->drivetrains[$value])
                ? $this->drivetrains[$value]
                : $this->getMappedTaxonomy(
                    TaxonomyMap::DRIVETRAIN,
                    $value,
                    $vehicle
                ),
            'transmission' => $value && !empty($this->transmissions[$value])
                ? $this->transmissions[$value]
                : $this->getMappedTaxonomy(
                    TaxonomyMap::TRANSMISSION,
                    $value,
                    $vehicle
                ),
            'featured' => $value ?? 0,
            default => $value,
        };
    }

    private function loadRelationships(): void
    {
        $this->dealerships = Dealership::all();
        $this->makes = VehicleMake::all()->pluck('id', 'name');
        $this->models = VehicleModel::all()->pluck('id', 'name');
        $this->fuel_types = FuelType::all()->pluck('id', 'name');
        $this->body_styles = BodyStyleType::all()->pluck('id', 'name');
        $this->drivetrains = DrivetrainType::all()->pluck('id', 'name');
        $this->transmissions = TransmissionType::all()->pluck('id', 'name');
    }

    protected function getProviderName(): string
    {
        if (empty($this->import_map)) {
            return $this->importType;
        }

        return 'import-map-' . $this->import_map->id;
    }

    /**
     * Get the dealership id from stock location
     *
     * @param $locationId
     * @return string|null
     */
    private function dealershipId($locationId): ?string
    {
        return $this->dealerships
            ->filter(fn(Dealership $dealership) => $dealership[$this->dealershipMatchValue] == $locationId
                || ($dealership->data[$this->dealershipMatchValue] ?? null) == $locationId)
            ->first()
            ?->id;
    }

    private function mapBasedOnImportMap(array $row): array
    {
        if (!isset($this->import_map)) {
            $this->import_map = ImportMap::query()->with('fields')->find($this->importType);
            if (!$this->import_map) {
                throw new Exception('Unrecognized import type: ' . $this->importType);
            }
        }

        if (!empty($this->import_map->data['dealership_field'])) {
            $this->dealershipMatchValue = $this->import_map->data['dealership_field'];
        }

        $data = $this->import_map->fields
            ->keyBy(fn(ImportMapField $field) => $field->on_model)
            ->filter(fn(ImportMapField $field, $key) => !empty($key))
            ->map(fn(ImportMapField $field) => $this->ensureFormattingForField(
                $field->on_model,
                $row[$field->on_file] ?? $this->decodeDefaultValue($field->default_value, $row),
                $row
            ))
            ->reject(fn($value) => is_null($value))
            ->toArray();
        return $this->verifyRowRequiredFields($data);
    }

    protected function verifyRowRequiredFields(array $data): array
    {
        $required_fields = $this->import_map->fields
            ->filter(fn(ImportMapField $field) => !empty($field->data['required_to_import']))
            ->pluck('on_model');

        $has_missing_required_fields = $required_fields
            ->filter(fn(string $field) => empty($data[$field]) || $this->rowFailsValidation($field, $data))
            ->isNotEmpty();

        return $has_missing_required_fields
            ? []
            : $data;
    }

    private function rowFailsValidation(string $field, array $data): bool
    {
        if ($field === 'vin') {
            return strlen($data[$field]) < 17;
        }

        return false;
    }

    private function ensureFormattingForField(mixed $on_model, $value, $row): mixed
    {
        $country = Settings::get('app-details-country') ?? config('app.default_country');
        return match ($on_model) {
            'registration_number' => Country::normalizeNumberPlate($country, $value ?? '', false),
            'make_id' => $this->getMappedTaxonomy(TaxonomyMap::MAKE, $value, $row),
            'model_id' => $this->getMappedTaxonomy(TaxonomyMap::MODEL, $value, $row), // TODO: Needs make value for map
            'transmission_id' => $this->getMappedTaxonomy(TaxonomyMap::TRANSMISSION, $value, $row),
            'body_style_id' => $this->getMappedTaxonomy(TaxonomyMap::BODY_STYLE, $value, $row),
            'drivetrain_id' => $this->getMappedTaxonomy(TaxonomyMap::DRIVETRAIN, $value, $row),
            'fuel_type_id' => $this->getMappedTaxonomy(TaxonomyMap::FUEL_TYPE, $value, $row),
            'type' => match (strtolower($value ?? '')) {
                'lcv' => VehicleType::LCV->value,
                'motorcycle' => VehicleType::MOTORCYCLE->value,
                default => VehicleType::CAR->value,
            },
            'first_registration_date' => Carbon::parse($value),
            'dealership_id' => $this->dealershipId($value),
            default => is_string($value) ? trim($value) : $value,
        };
    }

    private function decodeDefaultValue($default, array $row)
    {
        // If not logic block, return literal value
        if (!(is_string($default) && Str::startsWith($default, '{{') && Str::endsWith($default, '}}'))) {
            return $default;
        }

        // Strip logic identifiers
        $lookup = str_replace(['{{', '}}'], '', $default);
        $matched_operator = collect([
            '>=',
            '<=',
            '!=',
            '>',
            '<',
            '='
        ])
            ->filter(fn($operator) => stripos($lookup, $operator) !== false)
            ->first();

        // not a supported operator, return column with specified name or null if no column exists
        if (!$matched_operator) {
            return $row[$lookup] ?? null;
        }

        $conditions = explode($matched_operator, $lookup);
        // Perform a logic check based on left side being a column in file and right side being a value to check against
        return match ($matched_operator) {
            '>=' => ($row[$conditions[0]] ?? '') >= $conditions[1],
            '<=' => ($row[$conditions[0]] ?? '') <= $conditions[1],
            '!=' => ($row[$conditions[0]] ?? '') != $conditions[1],
            '>' => ($row[$conditions[0]] ?? '') > $conditions[1],
            '<' => ($row[$conditions[0]] ?? '') < $conditions[1],
            '=' => ($row[$conditions[0]] ?? '') == $conditions[1],
            default => $conditions[0],
        };
    }
}
