<?php

namespace Tests\Feature;

use App\Console\Commands\AutoTraderDataFetch;
use App\Facades\Settings;
use App\Jobs\FetchAutoTraderPerformanceData;
use Database\Seeders\Tenant\AutoTraderEnrichedDataSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Vehicle;
use Tests\TenantTestCase;
use Tests\TestCase;

class AutoTraderEnrichedDataFetchTest extends TenantTestCase
{
    public function testSendsJob()
    {
        $this->seed(AutoTraderEnrichedDataSeeder::class);
        Settings::update('autotrader-advert-performance', true);
        Queue::fake();
        $this->artisan(AutoTraderDataFetch::class)->assertExitCode(0);
        Queue::assertPushed(FetchAutoTraderPerformanceData::class);
    }

    public function testRunJob()
    {
        $this->seed(AutoTraderEnrichedDataSeeder::class);
        Settings::update('autotrader-advert-performance', true);
        $dealership = Dealership::factory()->create(['data' => ['auto-trader-location-id' => 100]]);
        $prefix = 'https://api-sandbox.autotrader.co.uk';
        $prefixLive = 'https://api-sandbox.autotrader.co.uk';
        /** @var Vehicle $vehicle */
        $vehicle = Vehicle::factory()->create([
            'uuid' => 100,
            'dealership_id' => $dealership->id,
        ]);
        Http::fake([
            $prefix . '/stock*' => $this->stockResponse(),
            $prefixLive . '/stock*' => $this->stockResponse(),
            '*' => Http::response(null, 400),
        ]);

        $this->artisan(AutoTraderDataFetch::class)->assertExitCode(0);

        $atData = $vehicle->autoTraderData;
        $this->assertNotNull($atData);
        $this->assertEquals('GOOD', $atData->price_point);
        $this->assertEquals(5000, $atData->valuation);
        $this->assertEquals(123, $atData->search_results_7d);
        $this->assertEquals(23, $atData->ad_views);
    }

    public function testBadResponse()
    {
        $this->seed(AutoTraderEnrichedDataSeeder::class);
        Settings::update('autotrader-advert-performance', true);
        Dealership::factory()->create(['data' => ['auto-trader-location-id' => 100]]);
        $prefix = 'https://api-sandbox.autotrader.co.uk';
        /** @var Vehicle $vehicle */
        $vehicle = Vehicle::factory()->create(['uuid' => 100]);
        Http::fake([
            $prefix . '/stock*' => $this->badResponse(),
            '*' => Http::response(null, 400),
        ]);

        $this->artisan(AutoTraderDataFetch::class)->assertExitCode(0);

    }

    private function stockResponse(): \GuzzleHttp\Promise\PromiseInterface
    {
        return Http::response([
            'results' => [
                [
                    'metadata' => [
                        'externalStockId' => 100,
                        'stockId' => 'abc123',
                    ],
                    'adverts' => [
                        'retailAdverts' => [
                            'priceIndicatorRating' => 'GOOD',
                        ]
                    ],
                    'vehicle' => [
                        'valuations' => [
                            'marketAverage' => [
                                'retail' => [
                                    'amountGBP' => 5000,
                                ]
                            ]
                        ]
                    ],
                    'responseMetrics' => [
                        'lastWeek' => [
                            'searchViews' => 123,
                            'advertViews' => 23,
                        ]
                    ]
                ],
            ],
            'totalResults' => 230
        ]);
    }

    private function badResponse(): \GuzzleHttp\Promise\PromiseInterface
    {
        return Http::response([
            'results' => [
            ],
        ]);
    }
}
