<?php

namespace App\Exports;

use App\Facades\Settings;
use App\IntegrationRepository;
use App\SalesChannelType;
use App\Services\Modix;
use App\TaxonomyMap;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Str;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\SalesChannelHistory;
use Mtc\MercuryDataModels\Tenant;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\Notifications\Facades\Notification;
use SimpleXMLElement;
use Exception;

class VolvoSelectExport
{
    use UsesSalesChannelRules;

    public function getData()
    {
        $data = [];
        Tenant::all()->each(function ($tenant) use (&$data) {
            tenancy()->initialize($tenant);
            if ((new IntegrationRepository())->isEnabled('volvo-select') !== true) {
                return;
            }

            try {
                $vehicles = $this->getVehicles();
                if ($vehicles->count()) {
                    $data[$tenant->id] = [
                        'dealerships' => $this->getDealerships(),
                        'vehicles' => $vehicles
                    ];

                    SalesChannelHistory::store('ebay-motors', true, $vehicles->count() . ' records exported');
                }

            } catch (Exception $exception) {
                dump($exception->getTrace());
                dd($exception->getMessage());
                SalesChannelHistory::store('ebay-motors', false, $exception->getMessage());
            }
        });

        return $data;
    }

    private function getDealerships()
    {
        return Dealership::query()
            ->where('active', 1)
            ->has('vehicles')
            ->get()
            ->map(function (Dealership $dealership) {
                return [
                    'id' => $dealership->data['volvo-select-dealer-id'] ?? null,
                    'name' => $dealership->name,
                    'city' => $dealership->city,
                    'zip' => $dealership->postcode,
                    'country' => $dealership->country,
                    'lat' => $dealership->lat,
                    'lng' => $dealership->lng,
                    'street' => $dealership->address1,
                    'phone' => $dealership->contact_no,
                    'email' => $dealership->email,
                ];
            })
            ->keyBy('id');
    }

    private function getVehicles()
    {
        return Vehicle::query()
            ->where('id', 309)
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'bodyStyle',
                'fuelType',
                'drivetrain',
                'specs',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::VOLVO_SELECT),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::VOLVO_SELECT)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'dealer_id' => $vehicle->dealership?->data['volvo-select-dealer-id'] ?? null,
                    'vehicle_id' => $vehicle->uuid,
                    'registration' => str_replace(' ', '', $vehicle->registration_number),
                    'make' => $this->getTaxonomy($vehicle, 'make'),
                    'model' => $this->getTaxonomy($vehicle, 'model', $vehicle->make_id),
                    'description' => htmlspecialchars($vehicle->description, ENT_XML1 | ENT_QUOTES),
                    'derivative' => htmlspecialchars($vehicle->derivative, ENT_XML1 | ENT_QUOTES),
                    'mileage' => $vehicle->odometer_mi,
                    'price' => $vehicle->price,
                    'type' => $vehicle->type,
                    'engine_size' => $vehicle->engine_size_cc,
                    'fuel_type' => $this->getTaxonomy($vehicle, 'fuelType'),
                    'drivetrain' => $this->getTaxonomy($vehicle, 'drivetrain'),
                    'transmission_type' => $this->getTaxonomy($vehicle, 'transmission'),
                    'manufacturer_year' => $vehicle->manufacture_year,
                    'registration_date' => $vehicle->first_registration_date?->format('d/m/Y'),
                    'vin' => $vehicle->vin,
                    'doors' => $vehicle->door_count,
                    'seats' => $vehicle->seats,
                    'body_style' => $this->getTaxonomy($vehicle, 'bodyStyle'),
                    'colour' => $vehicle->colour,
                    'picture_refs' => $vehicle->mediaUses
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large')),
                    'product_url' => UiUrlGenerator::make($vehicle->urlPath()),
                    'cap_id' => $vehicle->cap_id,
                    'is_new' => $vehicle->is_new,
                    'previous_owner_count' => $vehicle->previous_owner_count ?? null,
                    'options' => $vehicle->specs
                        ->pluck('description')
                        ->implode(','),
                ];
            })
            ->filter(fn($item) => !empty($item['dealer_id']))
            ->groupBy('dealer_id');
    }

    public function generateXml()
    {
        $data = $this->getData();

        $xml = new SimpleXMLElement('<?xml version="1.0" encoding="UTF-8"?><automotive version="3.0"></automotive>');
        foreach ($data as $tenant) {
            foreach ($tenant['vehicles'] as $dealershipId => $vehicles) {
                $dealership = $tenant['dealerships'][$dealershipId];

                $vehiclePool = $xml->addChild('vehiclePool');
                $vehiclePool->addAttribute('key', 'pool_' . $dealershipId);
                $vehiclePool->addAttribute('lang', 'en_GB');
                $dealer = $vehiclePool->addChild('dealer');
                $dealer->addAttribute('modkey', $dealershipId);
                $address = $dealer->addChild('address');
                $address->addChild('company', $dealership['name']);
                $address->addChild('brands');
                $address->addChild('street', $dealership['street']);
                $address->addChild('city', $dealership['city']);
                $address->addChild('zip', $dealership['zip']);
                $address->addChild('country', $dealership['country']);
                $coordinates = $address->addChild('coordinates');
                $coordinates->addChild('lat', $dealership['lat']);
                $coordinates->addChild('lon', $dealership['lng']);
                $address->addChild('phone', $dealership['phone']);
                $address->addChild('fax');
                $address->addChild('email', $dealership['email']);
                $address->addChild('vendorNumber');
                $address->addChild('imprint');
                $vehiclesNode = $vehiclePool->addChild('vehicles');
                foreach ($vehicles as $vehicle) {
                    $vehicleNode = $vehiclesNode->addChild('vehicle');
                    $vehicleNode->addAttribute('cipher', $vehicle['vehicle_id']);
                    $vehicleNode->addAttribute('lang', 'en_GB');
                    // Main Data
                    $mainData = $vehicleNode->addChild('mainData');

                    $this->addCategory($mainData, $vehicle['type']);
                    $this->addUsageCategory($mainData, $vehicle['is_new']);

                    $mainData->addChild('manufacturer', $vehicle['make']['term'])->addAttribute('id',
                        $vehicle['make']['modix_id']);
                    $mainData->addChild('model', $vehicle['model']['term'])
                        ->addAttribute('id', $vehicle['model']['modix_id']);
                    $mainData->addChild('modelText', $vehicle['model']['term']);
                    $mainData->addChild('submodel', $vehicle['derivative']);
                    $mainData->addChild('mileage', $vehicle['mileage'])
                        ->addAttribute('scaleUnit', 'mi');
                    $mainData->addChild('fuel', $vehicle['fuel_type']['term'])
                        ->addAttribute('id', $vehicle['fuel_type']['modix_id']);
                    $mainData->addChild('transmission', $vehicle['transmission_type']['term'])
                        ->addAttribute('id', $vehicle['transmission_type']['modix_id']);
                    $mainData->addChild('bodyStyle', $vehicle['body_style']['term'])
                        ->addAttribute('id', $vehicle['body_style']['modix_id']);
                    $mainData->addChild('isOnline', 'true');
                    $mainData->addChild('vehicleStatus', '100');

                    $prices = $vehicleNode->addChild('prices');
                    $price = $prices->addChild('price');
                    $price->addAttribute('currency', Settings::get('app-details-currency'));
                    $price->addAttribute('type', '2');
                    $price->addChild('grossPrice', $vehicle['price']);

                    $identification = $vehicleNode->addChild('identification');
                    $identification->addChild('vin', $vehicle['vin']);

                    $colors = $vehicleNode->addChild('colors');
                    $bodyColors = $colors->addChild('bodyColors');
                    $bodyColor = $bodyColors->addChild('bodyColor');
                    $bodyColor->addChild('name', $vehicle['colour']);
                    $bodyColor->addChild('description', $vehicle['colour']);
                    $bodyColor->addChild('paint', 'solid')
                        ->addAttribute('id', 1);

                    if (strtolower($vehicle['type']) == 'lcv') {
                        $price->addAttribute('vatReclaimable', 'true');
                    }

                    $vehicleNode->addChild('description', $vehicle['description']);

                    $options = $vehicleNode->addChild('options');
                    $option = $options->addChild('option');
                    $option->addAttribute('id', 1);
                    $option->addChild('value', $vehicle['doors']);
                    $option->addChild('name', $vehicle['doors']
                        . ' ' . Str::plural('door', $vehicle['doors']));
                    $option->addChild('description', 'Doors');

                    $option = $options->addChild('option');
                    $option->addAttribute('id', 2);
                    $option->addChild('value', $vehicle['seats']);
                    $option->addChild('name', $vehicle['seats']
                        . ' ' . Str::plural('seat', $vehicle['seats']));
                    $option->addChild('description', 'Seats');

                    if ($vehicle['previous_owner_count']) {
                        $option = $options->addChild('option');
                        $option->addAttribute('id', 14);
                        $option->addChild('value', $vehicle['previous_owner_count']);
                        $option->addChild('name', $vehicle['previous_owner_count']
                            . ' ' . Str::plural('previous owner', $vehicle['previous_owner_count']));
                        $option->addChild('description', 'Previous owner');
                    }

                    if (!empty($vehicle['drivetrain']['modix_id'])) {
                        $option = $options->addChild('option');
                        $option->addAttribute('id', 32);
                        $option->addChild('value', $vehicle['drivetrain']['modix_id']);
                        $option->addChild('name', $vehicle['drivetrain']['term']);
                        $option->addChild('description', 'Drive train');
                    }

                    if ($vehicle['manufacturer_year']) {
                        $option = $options->addChild('option');
                        $option->addAttribute('id', 336);
                        $option->addChild('value', $vehicle['manufacturer_year']);
                        $option->addChild('name');
                        $option->addChild('description', 'Model year');
                    }

                    $option = $options->addChild('option');
                    $option->addAttribute('id', 534);
                    $option->addChild('value', Settings::get('app-details-currency'));
                    $option->addChild('name');
                    $option->addChild('description', 'Currency');

                    if ($vehicle['options']) {
                        $option = $options->addChild('option');
                        $option->addAttribute('id', 500);
                        $option->addChild('value', $vehicle['options']);
                        $option->addChild('name');
                        $option->addChild('description', 'Special equipment');
                    }

                    if ($vehicle['picture_refs']->count()) {
                        $baseUrl = null;
                        $media = $vehicleNode->addChild('media');
                        $images = $media->addChild('images');

                        $vehicle['picture_refs']->each(function ($url, $index) use ($images, &$baseUrl) {
                            $pathParts = pathinfo($url);

                            if (!$baseUrl) {
                                $baseUrl = $pathParts['dirname'] . '/';
                                $images->addAttribute('url', $baseUrl);
                            }

                            $image = $images->addChild('image');
                            $image->addAttribute('name', $pathParts['basename']);
                            $image->addAttribute('position', $index + 1);
                        });
                    }
                }
                $vehiclePool->addChild('packages');
            }
        }

        // Return XML content as a string
        return $xml->asXML();
    }

    private function getTaxonomy($vehicle, $relation, ?int $makeId = null)
    {
        if (!empty($vehicle->{$relation})) {
            $morphClass = $vehicle->{$relation}->getMorphClass();
            $taxonomyMap = TaxonomyMap::searchTaxonomyByType(
                'modix',
                $morphClass == 'body_type' ? TaxonomyMap::BODY_STYLE : $morphClass,
                $vehicle->{$relation}->id,
                $makeId
            );

            if ($taxonomyMap) {
                return [
                    'id' => $taxonomyMap->id,
                    'modix_id' => $taxonomyMap['details']['modix_id'],
                    'term' => $taxonomyMap->term
                ];
            }

            ApiNotification::query()
                ->create([
                    'provider' => 'modix',
                    'data' => 'Taxonomy "' . $vehicle->{$relation}->name . '" search failed',
                    'data_model' => 'vehicle',
                    'reference' => $vehicle->id,
                ]);
        }

        return [
            'id' => '',
            'modix_id' => '',
            'term' => $vehicle->{$relation}?->name
        ];
    }

    private function addCategory($mainData, $type)
    {
        $category = match ($type) {
            VehicleType::CAR, VehicleType::LCV => ['id' => 1, 'name' => 'Passenger car'],
            VehicleType::MOTORCYCLE => ['id' => 5, 'name' => 'Motorbike'],
            default => null
        };

        if (!empty($category)) {
            $mainData->addChild('category', $category['name'])
                ->addAttribute('id', $category['id']);
        }
    }

    private function addUsageCategory($mainData, $isNew)
    {
        $usageCategory = $isNew ? [
            'id' => 1,
            'name' => 'New vehicle'
        ] : [
            'id' => 2,
            'name' => 'Used vehicle'
        ];

        $mainData->addChild('usageCategory', $usageCategory['name'])
            ->addAttribute('id', $usageCategory['id']);
    }
}
