<?php

namespace App;

use App\Contracts\AbleToSyncContentElements;
use App\Contracts\InteractsWithContentSync;
use App\Facades\Settings;
use App\Modules\Lookup\Contracts\VehicleLookupData;
use App\Traits\EnsuresVehicleAttribute;
use App\Traits\MapsTaxonomies;
use App\Traits\RetrievesFieldData;
use App\Traits\SavesContent;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Mtc\ContentManager\Facades\Media;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\VehicleTrim;

class NewCarRepository implements InteractsWithContentSync, AbleToSyncContentElements
{
    use RetrievesFieldData;
    use SavesSeoData;
    use SavesContent;
    use MapsTaxonomies;
    use EnsuresVehicleAttribute;

    protected static array $dataFields = [
        'make_id',
        'model_id',
        'template_id',
        'type_id',
        'form_id',
        'derivative',
        'price',
        'deposit',
        'seo',
        'franchise_id',
        'engine_size_cc',
        'colour',
    ];

    /**
     * Save Page
     *
     * @param NewCar $car
     * @param Request $request
     * @param int $author
     * @return void
     */
    public function save(NewCar $car, Request $request, int $author)
    {
        DB::beginTransaction();

        $this->dropOrphanedContent($car, $this->getFlatContentElementIds($request->input('content')));
        collect($request->input('content'))
            ->each(fn($element, $index) => $this->saveContentElement($car, $element, $index));

        $car->fill($request->input());
        $car->slug = $request->input('seo.slug') ?? $car->slug;
        $car->save();

        if ($request->filled('seo')) {
            $this->saveSeo($car, $request->input('seo', []));
        }

        Media::setUsesForModel(
            $request->input('listing_image', []),
            $car,
            ['flags' => ['listing' => true]],
            false
        );
        Media::setUsesForModel(
            $request->input('hero_image', []),
            $car,
            ['flags' => ['hero' => true]],
            false
        );
        Media::setUsesForModel(
            $request->input('mobile_hero_image', []),
            $car,
            ['flags' => ['mobile_hero' => true]],
            false
        );

        $allMedia = array_merge(
            $request->input('listing_image', []),
            $request->input('hero_image', []),
            $request->input('mobile_hero_image', [])
        );
        Media::removeMediaUsesNotInList($car, $allMedia);

        $car->fuelTypes()->sync($request->input('fuel_types'));

        collect($request->input('extra', []))
            ->filter(fn($attribute) => isset($attribute['value']) && empty($attribute['readonly']))
            ->each(fn($attribute) => $this->saveAttributeValue($car, $attribute));

        if ($this->hasVersioning()) {
            $this->saveVersion($car, $request->input('content', []), $author);
        }
        DB::commit();
    }

    public function copy(NewCar $origin, Request $request): NewCar
    {
        $origin->load([
            'content.subContent.subContent.subContent',
        ]);

        /** @var NewCar $copy */
        $copy = NewCar::query()->create($origin->toArray());

        if ($request->input('with_content')) {
            $this->copyContent($copy, $origin->content);
        }

        return $copy;
    }


    public function hasVersioning(): bool
    {
        return Settings::get('new-cars-versioning-enabled') === true
            && TierHelper::isAllowed(tenant('tier'), Tier::STANDARD->value);
    }

    public function mapTechnicalData(VehicleLookupData $data): array
    {
        return [
            'derivative' => $data->derivative,
            'engine_size_cc' => $data->engine_capacity_cc,
            'drivetrain_id' => $this->getMappedTaxonomy(
                TaxonomyMap::DRIVETRAIN,
                $data->drivetrain,
                $data->toArray()
            ),
            'body_style_id' => $this->getMappedTaxonomy(
                TaxonomyMap::BODY_STYLE,
                $data->body_type,
                $data->toArray()
            ),
            'transmission_id' => $this->getMappedTaxonomy(
                TaxonomyMap::TRANSMISSION,
                $data->transmission,
                $data->toArray()
            ),
            'fuel_type' => $this->getMappedTaxonomy(
                TaxonomyMap::FUEL_TYPE,
                $data->fuel_type,
                $data->toArray()
            )
        ];
    }

    protected function elementParentColumnName(): string
    {
        return 'car_id';
    }

    protected function contentOwnerType(): string
    {
        return 'new-car-version';
    }

    protected function getQuery(): Builder
    {
        return NewCar::query();
    }

    public function importRecord(array $entry): bool
    {
        $newCar = $this->getQuery()
            ->create(Arr::only($entry, (new NewCar())->getFillable()));

        if (!empty($entry['hero_image'])) {
            $media = Media::importImageFromUrl($entry['hero_image']);
            Media::setUsesForModel([$media->id], $newCar, ['flags' => ['listing' => true]], false);
        }

        if (!empty($entry['listing_image'])) {
            $media = Media::importImageFromUrl($entry['listing_image']);
            Media::setUsesForModel([$media->id], $newCar, ['flags' => ['hero' => true]], false);
        }

        if (!empty($entry['fuel_types'])) {
            $newCar->fuelTypes()->sync($entry['fuel_types']);
        }

        return $newCar->exists;
    }

    public function canBeImported(array $entry): bool
    {
        return !$this->getQuery()
            ->where('slug', $entry['slug'])
            ->exists();
    }

    public function exportToRemote(array $selections): array
    {
        return $this->getQuery()
            ->with('mediaUses.media', 'fuelTypes:id')
            ->whereIn('id', $selections)
            ->get()
            ->map(function ($newCar) {
                $newCarData = $newCar->toArray();

                $newCarData['hero_image'] = $newCar->mediaUses
                    ->filter(fn($mediaUse) => $mediaUse->flags['hero'] ?? null)
                    ->first()
                    ?->media
                    ->getOriginalUrlAttribute();

                $newCarData['listing_image'] = $newCar->mediaUses
                    ->filter(fn($mediaUse) => $mediaUse->flags['listing'] ?? null)
                    ->first()
                    ?->media
                    ->getOriginalUrlAttribute();

                unset($newCarData['media_uses']);

                $newCarData['fuel_types'] = $newCar->fuelTypes->pluck('id')->toArray();

                return $newCarData;
            })
            ->toArray();
    }

    public function checkImportEntryValidity(array $dataEntry, array $allEntries): array
    {
        $errors = [];

        if (empty($dataEntry['slug'])) {
            $errors[] = __('validation.import_slug_missing', ['slug' => $dataEntry['slug']]);
        } elseif ($this->getQuery()->where('slug', $dataEntry['slug'])->exists()) {
            $errors[] = __('validation.import_slug_taken', ['slug' => $dataEntry['slug']]);
        }

        return [
            'data' => $dataEntry,
            'errors' => $errors,
        ];
    }

    /**
     * @param NewCar $car
     * @param array $data
     * @return void
     * @throws ValidationException
     */
    public function saveTrims(NewCar $car, array $data): void
    {
        collect($data)->each(function ($trim, $order) use ($car) {
            $existing_trim = VehicleTrim::query()
                ->where('make_id', '=', $car->make_id)
                ->where('model_id', '=', $car->model_id)
                ->where('name', '=', $trim['name'])
                ->first();

            if (
                $existing_trim
                && (
                    !array_key_exists('id', $trim)
                    || $existing_trim->id != $trim['id'])
            ) {
                // we are trying to add a new trim but the name is a duplicate of an existing trim
                throw ValidationException::withMessages(['Duplicate trim name: ' . $trim['name']]);
            } else {
                if (array_key_exists('id', $trim)) {
                    $existing_trim = VehicleTrim::find($trim['id']);
                }

                // either a trim with this name does not yet exist
                // or we are updating the trim with this name
                $trim_model = $existing_trim ?? VehicleTrim::query()->create([
                    'make_id' => $car->make_id,
                    'model_id' => $car->model_id,
                    'name' => $trim['name'] ?? null,
                ]);
            }

            $trim_model->fill([
                'order' => $order,
                'make_id' => $car->make_id,
                'model_id' => $car->model_id,
                'name' => $trim['name'] ?? null,
                'fuel_types' => $trim['fuel_types'] ?? null,
                'description' => $trim['description'] ?? null,
                'colours' => $trim['colours'] ?? null,
                'transmission_types' => $trim['transmission_types'] ?? null,
                'mpg' => $trim['mpg'] ?? null,
                'emissions' => $trim['emissions'] ?? null,
                'engine_size' => $trim['engine_size'] ?? null,
                'battery_size' => $trim['battery_size'] ?? null,
                'range_miles' => $trim['range_miles'] ?? null,
                'bhp' => $trim['bhp'] ?? null,
                'active' => $trim['active'] ?? null,
                'zero_to_sixty' => $trim['zero_to_sixty'] ?? null,
                'doors' => $trim['doors'] ?? null,
                'seats' => $trim['seats'] ?? null,
                'battery_charge_time' => $trim['battery_charge_time'] ?? null,
                'battery_quick_charge_time' => $trim['battery_quick_charge_time'] ?? null,
                'wheelbase' => $trim['wheelbase'] ?? null,
                'cap_id' => $trim['cap_id'] ?? null,
                'full_price' => $trim['full_price'] ?? null,
            ])->save();

            if (empty($trim['image'])) {
                $trim_model->mediaUses()->where('primary', true)->delete();
            } else {
                $trim_model->mediaUses()
                    ->where('primary', true)
                    ->whereNotIn('media_id', $trim['image'])
                    ->delete();

                Media::setUsesForModel(
                    $trim['image'],
                    $trim_model,
                    ['primary' => true],
                    false
                );
            }

            if (empty($trim['listing_image'])) {
                $trim_model->mediaUses()->where('primary', false)->delete();
            } else {
                $trim_model->mediaUses()
                    ->where('primary', false)
                    ->whereNotIn('media_id', $trim['listing_image'])
                    ->delete();

                Media::setUsesForModel(
                    $trim['listing_image'],
                    $trim_model,
                    ['flags' => ['listing' => true]],
                    false
                );
            }
        });
    }

    protected function getProviderName(): string
    {
        return 'new-car-repository';
    }
}
