<?php

namespace App\Console\Commands;

use App\Mail\NewVehicleMakesModelsMail;
use App\Master\Services\AutoTraderTaxonomyService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;

class SyncAutoTraderTaxonomies extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'auto-trader:sync-taxonomies';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Pull Taxonomy data from AutoTrader (makes/models)';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(protected AutoTraderTaxonomyService $service)
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $makes = collect($this->service->getMakes())
            ->filter(fn($make) => !empty($make['name']) && !Str::contains($make['name'], ["'", '+', '!', '=', '.']))
            ->map(function (array $make) {
                return VehicleMake::query()
                    ->updateOrCreate(
                        ['autotrader_id' => $make['makeId']],
                        ['name' => $make['name']]
                    );
            });

        $this->info($makes->count() . ' fetched');
        $allMakes = VehicleMake::query()->pluck('id', 'autotrader_id');

        $makes->each(fn(VehicleMake $make) => $this->syncModels($make['autotrader_id'], $allMakes));
        $this->notifyOnAdditions();

        return 0;
    }

    /**
     * Sync model data for a make
     *
     * @param $makeId
     * @param $all_makes
     */
    protected function syncModels($makeId, Collection $allMakes)
    {
        if (empty($allMakes[$makeId])) {
            return;
        }

        collect($this->service->getModels($makeId))
            ->filter(fn($model) => !empty($model['name']))
            ->each(function (array $model) use ($makeId, $allMakes) {
                // For some reason AT has default A6 as A6 Unspecified
                if ($model['name'] == 'A6 Unspecified') {
                    $model['name'] = 'A6';
                }
                $existing = VehicleModel::query()
                    ->where('autotrader_id', $model['modelId'])
                    ->first();
                if (!$existing) {
                    VehicleModel::query()->create([
                        'autotrader_id' => $model['modelId'],
                        'name' => $model['name'],
                        'make_id' => $allMakes[$makeId]
                    ]);
                } elseif (!$existing->do_not_sync) {
                    $existing->update([
                        'name' => $model['name'],
                        'make_id' => $allMakes[$makeId],
                    ]);
                }
            });
    }

    /**
     * Notify on new additions
     *
     * @return void
     */
    private function notifyOnAdditions()
    {
        $recentMakes = VehicleMake::query()
            ->where('created_at', '>', Carbon::now()->startOfDay())
            ->get();

        $recentModels = VehicleModel::query()
            ->where('created_at', '>', Carbon::now()->startOfDay())
            ->get();

        if ($recentMakes->isEmpty() && $recentModels->isEmpty()) {
            return;
        }

        Mail::to(config('mail.developers'))
            ->send(new NewVehicleMakesModelsMail($recentMakes, $recentModels));
    }
}
