<?php

namespace App\Console\Commands;

use App\Facades\Settings;
use Illuminate\Console\Command;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\DealershipDepartment;

class PopulateDealershipDepartments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'dealerships:populate-departments';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create departments based on dealership opening times';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        Dealership::query()
            ->get()
            ->each(function ($dealership) {
                $this->createDepartmentsForDealership($dealership);
            });
        return 0;
    }

    private function createDepartmentsForDealership(Dealership $dealership): void
    {
        if (
            DealershipDepartment::query()
            ->where('dealership_id', '=', $dealership->id)
            ->count()
        ) {
            // don't populate departments for dealerships that already have departments
            return;
        }

        if (!empty($dealership->open_times)) {
            $this->createDealershipDepartment(
                $dealership->id,
                $dealership->open_times,
                'General',
                $dealership->email,
                $dealership->contact_no,
                true
            );
        }

        if (!empty($dealership->alt_open_times)) {
            $this->createDealershipDepartment(
                $dealership->id,
                $dealership->alt_open_times,
                !empty(Settings::get('dealership-secondary-open-close-time-name'))
                    ? Settings::get('dealership-secondary-open-close-time-name')
                    : 'Department Two',
                $dealership->email,
                $dealership->contact_no,
            );
        }

        if (!empty($dealership->second_alt_open_times)) {
            $this->createDealershipDepartment(
                $dealership->id,
                $dealership->second_alt_open_times,
                !empty(Settings::get('dealership-tertiary-open-close-time-name'))
                    ? Settings::get('dealership-tertiary-open-close-time-name')
                    : 'Department Three',
                $dealership->email,
                $dealership->contact_no,
            );
        }
    }

    private function createDealershipDepartment(
        int $dealership_id,
        array $open_times,
        string $name,
        ?string $email,
        ?string $contact_no,
        bool $is_primary = false
    ) {
        DealershipDepartment::query()->create([
            'dealership_id' => $dealership_id,
            'name' => $name,
            'open_times' => $open_times,
            'email' => $email,
            'contact_no' => $contact_no,
            'is_primary' => $is_primary,
        ]);
    }
}
