<?php

namespace Tests\Feature;

use App\Facades\Settings;
use App\Jobs\SyncContentToTemplate;
use App\PageRepository;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Queue;
use Mtc\MercuryDataModels\Page;
use Mtc\MercuryDataModels\Template;
use Mtc\MercuryDataModels\VehicleOffer;
use Tests\TestCase;
use Tests\UserForTenant;

class TemplateControllerTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    protected $tenancy = true;

    public function testIndex()
    {
        Template::factory(5)->create();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('content.templates.index'));

        $response->assertStatus(200);
        $this->assertCount(5, $response->json('data'));
    }

    public function testShow()
    {
        $template = Template::factory()->create();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('content.templates.show', $template));

        $response->assertStatus(200);
        $this->assertArrayHasKey('template', $response->json());
        $this->assertArrayHasKey('global_content', $response->json());
        $this->assertArrayHasKey('elements', $response->json());
        $this->assertArrayHasKey('comments', $response->json());
    }

    public function testUsage()
    {
        $template = Template::factory()->create();
        $pages = Page::factory(3)->create(['template_id' => $template->id]);
        Page::factory(3)->create(['template_id' => null]);
        $offers = VehicleOffer::factory(4)->create(['template_id' => $template->id]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.templates.usage', $template));

        $response->assertStatus(200);
        $this->assertCount(7, $response->json());
        $pages->each(fn($page) => $this->assertTrue(collect($response->json())->where('type', 'page')->where('id', $page->id)->isNotEmpty()));
        $offers->each(fn($offer) => $this->assertTrue(collect($response->json())->where('type', 'offer')->where('id', $offer->id)->isNotEmpty()));
    }

    public function testSync()
    {
        Queue::fake();
        $template = Template::factory()->create();
        $page = Page::factory()->create(['template_id' => $template->id]);
        Page::factory(3)->create(['template_id' => null]);
        $offer = VehicleOffer::factory()->create(['template_id' => $template->id]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.templates.sync-usage', $template));
        $response->assertStatus(200);
        Queue::assertPushed(SyncContentToTemplate::class);

    }

    public function testSyncProcess()
    {
        $template = Template::factory()->create();
        $page = Page::factory()->create(['template_id' => $template->id]);
        Page::factory(3)->create(['template_id' => null]);
        $offer = VehicleOffer::factory()->create(['template_id' => $template->id]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.templates.sync-usage', $template));
        $response->assertStatus(200);
    }

}
