<?php

namespace App\Providers;

use App\Master\Models\BodyStyleType;
use App\Master\Models\DrivetrainType;
use App\Master\Models\FuelType;
use App\Master\Models\TransmissionType;
use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;
use App\Modules\Stock\Motordat;
use App\Modules\Stock\Motors;
use App\Modules\VehicleLabels\LabelRepository;
use App\Observers\DealershipContentObserver;
use App\Observers\DealershipObserver;
use App\Observers\NewCarObserver;
use App\Observers\ReservationObserver;
use App\Observers\VehicleAttributeObserver;
use App\Observers\VehicleObserver;
use App\Observers\VehicleOfferContentObserver;
use App\Observers\VehicleOfferObserver;
use App\ReportRepository;
use App\Services\AutoTraderApi;
use App\SettingRepository;
use App\SiteRepository;
use Illuminate\Auth\Notifications\ResetPassword;
use Illuminate\Auth\Notifications\VerifyEmail;
use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\ServiceProvider;
use Laravel\Telescope\TelescopeApplicationServiceProvider;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\DealershipContent;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Repositories\FeatureRepository;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleAttribute;
use Mtc\MercuryDataModels\VehicleOffer;
use Mtc\MercuryDataModels\VehicleOfferContent;
use Mtc\VehicleReservations\Reservation;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        $this->setMigrationLocation();

        if (class_exists(TelescopeApplicationServiceProvider::class)) {
            $this->app->register(TelescopeServiceProvider::class);
        }
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        setPermissionsTeamId('');

        $this->overwriteCallbackUrls();

        $this->app->bind(AutoTraderApi::class, fn() => new AutoTraderApi(config('services.auto-trader', [])));
        $this->app->bind('site-settings', SettingRepository::class);
        $this->app->bind('reports', ReportRepository::class);
        $this->app->bind('site-repository', SiteRepository::class);
        $this->app->bind('label-repository', LabelRepository::class);
        $this->app->bind('feature-flags', FeatureRepository::class);

        $this->app->bind('motordat', Motordat::class);
        $this->app->bind('Motors', Motors::class);

        Vehicle::observe(VehicleObserver::class);
        NewCar::observe(NewCarObserver::class);
        Dealership::observe(DealershipObserver::class);
        VehicleOffer::observe(VehicleOfferObserver::class);
        Reservation::observe(ReservationObserver::class);
        VehicleOfferContent::observe(VehicleOfferContentObserver::class);
        DealershipContent::observe(DealershipContentObserver::class);
        VehicleAttribute::observe(VehicleAttributeObserver::class);
        Relation::morphMap([
            'master-make' => VehicleMake::class,
            'master-model' => VehicleModel::class,
            'master-fuel_type' => FuelType::class,
            'master-transmission' => TransmissionType::class,
            'master-body_type' => BodyStyleType::class,
            'master-body_style' => BodyStyleType::class,
            'master-drivetrain' => DrivetrainType::class,
        ]);

        // Model::preventLazyLoading(!$this->app->isProduction());

        RateLimiter::for('sync', fn($job) => Limit::perMinute('30'));

        if (!$this->app->environment('local') && !$this->app->environment('production')) {
            Mail::alwaysTo(config('mail.staging_log_email'));
        }
    }

    private function overwriteCallbackUrls()
    {
        ResetPassword::$createUrlCallback = function ($notifiable, $token) {
            return UiUrlGenerator::make('reset-password', [
                'token' => $token,
                'email' => $notifiable->email,
            ]);
        };

        VerifyEmail::$createUrlCallback = function ($notifiable, $token) {
            return UiUrlGenerator::make('email/verify' . $notifiable->getKey()
                . '/' . sha1($notifiable->getEmailForVerification()));
        };
    }

    /**
     * Due to the shared data models migrations are moved in mtcmedia/mercury-data-models
     *
     * @return void
     */
    private function setMigrationLocation()
    {
        $this->app->useDatabasePath($this->app->basePath('vendor/mtcmedia/mercury-data-models/database'));
        Config::set(
            'tenancy.migration_parameters.--path',
            $this->app->basePath('vendor/mtcmedia/mercury-data-models/database/migrations/tenant')
        );
    }
}
