<?php

namespace App\Services;

use App\Contracts\AIContentDriver;
use App\Facades\Settings;
use App\Master\Models\VehicleMake;
use App\Master\Models\VehicleModel;
use App\Models\EVData;
use App\Models\ExternalServiceUsage;
use App\VehicleType;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;

class EVDatabase
{
    public function sync(): bool
    {
        try {
            $this->saveCollection($this->getData('m1/bev'), VehicleType::CAR->value);
            $this->saveCollection($this->getData('m1/phev'), VehicleType::CAR->value);
            $this->saveCollection($this->getData('n1/bev'), VehicleType::LCV->value);
        } catch (\Exception $exception) {
            dump($exception);
            Log::error('Failed to retrieve EV Database info: ' . $exception->getMessage());
            return false;
        }
        return true;
    }

    public function match(Vehicle $vehicle): void
    {
        $data = EVData::query()
            ->where('make_id', $vehicle->make_id)
            ->where('model_id', $vehicle->model_id)
            ->when(
                $vehicle->first_registration_date,
                fn($query) => $query->where('date_from', '<=', $vehicle->first_registration_date)
                    ->where(fn($endQuery) => $endQuery->whereNull('date_to')
                        ->orWhere('date_to', '>=', $vehicle->first_registration_date))
            )
            ->first();

        if (!$data) {
            // match not found
            return;
        }

        dump($data);

    }

    private function getData(string $type): Collection
    {
        $response = Http::get($this->endpoint($type));
        if ($response->successful()) {
            return collect($response->json());
        }
        Log::error('Failed to get EV Database data for ' . $type, [
            $response->body()
        ]);
        return collect();
    }

    private function saveCollection(Collection $data, string $type): void
    {
        $data->each(function ($entry) use ($type) {
            /** @var EVData $record */
            $record = EVData::query()->firstOrNew(['ev_db_id' => $entry['Vehicle_ID']]);
            $record->fill([
                'vehicle_type' => $type,
                'variant' => $entry['Vehicle_Model_Version'] ?? null,
                'date_from' => isset($entry['Availability_Date_From'])
                    ? Carbon::createFromFormat('m-Y', $entry['Availability_Date_From'])->startOfMonth()
                    : null,
                'date_to' => isset($entry['Availability_Date_To'])
                    ? Carbon::createFromFormat('m-Y', $entry['Availability_Date_To'])->endOfMonth()
                    : null,
                'data' => $entry,
            ]);
            if (empty($record->make_id)) {
                $record->make_id = VehicleMake::query()
                    ->where('name', $entry['Vehicle_Make'])
                    ->first()
                    ?->id;
            }
            if (empty($record->model_id)) {
                $record->model_id = VehicleModel::query()
                    ->where('make_id', $record->make_id)
                    ->where('name', $entry['Vehicle_Model'])
                    ->first()
                    ?->id;
            }

            $record->save();
        });
    }

    private function endpoint(string $type): string
    {
        $api_user = Config::get('services.ev_data.user_id');
        $api_key = Config::get('services.ev_data.key');
        return "https://ev-database.org/export_v31/$type/$api_user/$api_key";

    }

}
