<?php

namespace Tests\Feature;

use App\Facades\Settings;
use App\IntegrationRepository;
use App\Tier;
use Database\Seeders\Tenant\ChatSettingSeeder;
use Database\Seeders\Tenant\FinanceSettingSeeder;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TenantTestCase;
use Tests\TestCase;

class IntegrationRepositoryTest extends TenantTestCase
{
    public function testGetEnabled()
    {
        $this->seed(FinanceSettingSeeder::class);
        $this->seed(ChatSettingSeeder::class);
        $repository = new IntegrationRepository();
        $enabled = $repository->getEnabled();
        $this->assertCount(0, $enabled);

        Settings::update('chat-rapid-rtc-enabled', true);
        Settings::update('finance-codeweavers-enabled', true);
        $enabled = $repository->getEnabled();
        $this->assertCount(2, $enabled);

    }

    public function testKeyIsEnableSettingValue()
    {
        $tests = [
            'sales-channels-motors-co-uk-enabled' => true,
            'sales-channels-motors-co-uk-ftp_username' => false,
            'crm-enquiry-max-enabled' => true,
            'chat-rapid-rtc-enabled' => true,
            'chat-live-chat-enabled' => true,
            'chat-live-chat-client_id' => false,
            'foo-baz-bar' => false,
        ];
        $repository = new IntegrationRepository();
        foreach ($tests as $value => $result) {
            $this->assertEquals($result, $repository->keyIsEnableSettingValue($value));
        }
    }

    public function testGetTypeFromKey()
    {
        $tests = [
            'sales-channels-motors-co-uk-enabled' => 'sales-channels',
            'crm-enquiry-max-enabled' => 'crm',
            'chat-rapid-rtc-enabled' => 'chat',
            'chat-live-chat-enabled' => 'chat',
            'foo-baz-bar' => null,
        ];
        $repository = new IntegrationRepository();
        foreach ($tests as $value => $result) {
            $this->assertEquals($result, $repository->getTypeFromKey($value));
        }
    }

    public function testGetLimitForType()
    {
        $repository = new IntegrationRepository();
        $this->assertEquals(2, $repository->getLimitForType('sales-channels', Tier::LITE->value));
        $this->assertEquals(5, $repository->getLimitForType('sales-channels', Tier::STANDARD->value));
        $this->assertEquals(999, $repository->getLimitForType('sales-channels', Tier::PRO->value));
        $this->assertEquals(1, $repository->getLimitForType('crm', Tier::LITE->value));
        $this->assertEquals(1, $repository->getLimitForType('chat', Tier::LITE->value));
    }

    public function testCheckEnabledCountForType()
    {
        $repository = new IntegrationRepository();
        $input = [
            'chat-rapid-rtc-enabled' => true,
            'chat-rapid-rtc-key' => 'foo',
            'chat-live-chat-enabled' => true,
            'chat-live-chat-client_id' => 'something',
        ];

        $this->assertEquals(2, $repository->checkEnabledCountForType($input, 'chat'));
        $input = [
            'chat-rapid-rtc-enabled' => false,
            'chat-rapid-rtc-key' => 'foo',
            'chat-live-chat-enabled' => true,
            'chat-live-chat-client_id' => 'something',
        ];

        $this->assertEquals(1, $repository->checkEnabledCountForType($input, 'chat'));
    }
}
