<?php

namespace App\Http\Requests;

use App\Rules\HasNoDuplicateFeatures;
use App\Rules\UniqueVehicleRegistrationNumber;
use App\Rules\ValidVehicleRegistrationNumber;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Mtc\MercuryDataModels\SeoData;

class UpdateVehicleRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'vin' => 'sometimes',
            'title' => 'required',
            'derivative' => 'required',
            'registration_number' => [
                'required_unless:is_new,true',
                new UniqueVehicleRegistrationNumber($this->vehicle?->id),
                new ValidVehicleRegistrationNumber(
                    $this->input('personalized_number_plate', false),
                    $this->input('is_new', false)
                )
            ],
            'is_new' => 'sometimes',
            'is_demo' => 'sometimes',
            'is_sold' => 'sometimes',
            'is_published' => [
                'present',
                'boolean',
            ],
            'make_id' => [
                'required',
                'numeric'
            ],
            'model_id' => [
                'required',
                'numeric'
            ],
            'dealership_id' => [
                'nullable',
                'numeric'
            ],
            'transmission_id' => [
                'nullable',
                'numeric'
            ],
            'fuel_type_id' => [
                'nullable',
                'numeric'
            ],
            'drivetrain_id' => [
                'nullable',
                'numeric'
            ],
            'colour' => [
                'nullable',
            ],
            'price' => [
                'nullable',
                'numeric'
            ],
            'original_price' => [
                'nullable',
                'numeric'
            ],
            'monthly_price' => [
                'nullable',
                'numeric'
            ],
            'rrp_price' => [
                'nullable',
                'numeric'
            ],
            'admin_fee' => [
                'nullable',
                'numeric'
            ],
            'previous_owner_count' => [
                'sometimes',
                'numeric',
            ],
            'deposit' => [
                'nullable',
                'numeric'
            ],
            'door_count' => [
                'nullable',
                'numeric'
            ],
            'seats' => [
                'nullable',
                'numeric'
            ],
            'manufacture_year' => [
                'nullable',
                'numeric'
            ],
            'odometer_km' => [
                'nullable',
                'numeric'
            ],
            'odometer_mi' => [
                'nullable',
                'numeric'
            ],
            'engine_size_cc' => [
                'nullable',
                'numeric'
            ],
            'co2' => [
                'nullable',
                'numeric'
            ],
            'mpg' => [
                'nullable',
                'numeric'
            ],
            'first_registration_date' => [
                'nullable',
                'date'
            ],
            'description' => [
                'sometimes',
            ],
            'media' => [
                'array'
            ],
            'media.*' => [
                'numeric',
                'exists:media,id'
            ],
            'primary_media' => [
                'present'
            ],
            'features' => [
                new HasNoDuplicateFeatures()
            ],
            'equipment' => 'array',
            'equipment.*.code' => 'sometimes',
            'equipment.*.description' => [
                'sometimes',
                'required'
            ],
            'standard_equipment.*.type' => [
                'required',
            ],
            'standard_equipment.*.price' => [
                'nullable',
                'numeric',
            ],
            'specs' => 'array',
            'specs.*.code' => 'sometimes',
            'specs.*.category' => 'sometimes',
            'specs.*.description' => [
                'required'
            ],
            'specs.*.value' => [
                'required'
            ],
            'seo.jsonSchema' => 'json',
            'seo.title' => [
                'nullable',
                'min:' . SeoData::TITLE_MIN_LENGTH,
                'max:' . SeoData::TITLE_MAX_LENGTH,
            ],
            'seo.description' => [
                'nullable',
                'min:' . SeoData::DESCRIPTION_MIN_LENGTH,
                'max:' . SeoData::DESCRIPTION_MAX_LENGTH,
            ],
        ];
    }

    public function attributes()
    {
        return [
            'seo.jsonSchema' => 'JSON schema',
            'is_new' => 'new vehicle',
            'make_id' => 'vehicle make',
            'model_id' => 'vehicle model',
            'dealership_id' => 'dealership',
            'transmission_id' => 'transmission',
            'fuel_type_id' => 'fuel type',
            'drivetrain_id' => 'drivetrain',
            'seo.title' => 'SEO page title',
            'seo.description' => 'SEO page description',
        ];
    }

    public function primaryMedia()
    {
        if (!empty($this->input('primary_media'))) {
            return $this->input('primary_media');
        }

        return collect($this->input('media', []))
            ->flip()
            ->first();
    }
}
