<?php

namespace App\Exports;

use App\SalesChannelType;
use App\Traits\UsesSalesChannelRules;
use App\VehicleType;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Vehicle;

class HeyCarExport implements FromCollection, WithHeadings
{
    use UsesSalesChannelRules;

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return Vehicle::query()
            ->with([
                'mediaUses.media',
                'dealership',
                'make',
                'model',
                'transmission',
                'fuelType',
                'bodyStyle',
            ])
            ->exportable()
            ->when(
                $this->hasSalesChannelRules(SalesChannelType::HEY_CAR),
                fn($query) => $this->applyQueryConditionsForChannel($query, SalesChannelType::HEY_CAR)
            )
            ->get()
            ->map(function (Vehicle $vehicle) {
                return [
                    'dealer_id' => $vehicle->dealership?->data['hey-car-dealer-id'],
                    'vehicle_id' => $vehicle->uuid,
                    'registration' => $vehicle->registration_number,
                    'vin' => $vehicle->vin,
                    'mileage' => $vehicle->odometer_mi,
                    'price' => $vehicle->price,
                    'recommended_retail_price' => $vehicle->price,
                    'vat_included' => $vehicle->vehicle_type === VehicleType::CAR->value ? 1 : 0,
                    'comments' => '',
                    'manufacturer_approved' => 0,
                    'retailer_approved' => 1,
                    'images' => $vehicle->mediaUses
                        ->map(fn(MediaUse $mediaUse) => $mediaUse->getUrl('large'))
                        ->implode(','),
                    'video' => '',
                    'stock_type' => !empty($vehicle->is_new) ? 'yes' : 'used',
                    'make' => $vehicle->make?->name,
                    'model' => $vehicle->model?->name,
                    'variant' => $vehicle->derivative,
                    'colour' => $vehicle->colour,
                    'body' => $vehicle->bodyStyle?->name,
                    'transmission' => $vehicle->transmission?->name,
                    'fuel_type' => $vehicle->fuelType?->name,
                    'year' => $vehicle->manufacture_year,
                    'cap_code' => $vehicle->cap_code,
                    'cap_id' => $vehicle->cap_id,
                    'doors' => $vehicle->door_count,
                    'engine_size' => $vehicle->engine_size_cc,
                ];
            });
    }

    /**
     * Heading row
     *
     * @return string[]
     */
    public function headings(): array
    {
        return [
            'dealer_id',
            'vehicle_id',
            'registration',
            'vin',
            'mileage',
            'price',
            'recommended_retail_price',
            'vat_included',
            'comments',
            'manufacturer_approved',
            'retailer_approved',
            'images',
            'video',
            'stock_type',
            'make',
            'model',
            'variant',
            'colour',
            'body',
            'transmission',
            'fuel_type',
            'year',
            'cap_code',
            'cap_id',
            'doors',
            'engine_size',
        ];
    }
}
