<?php

namespace Tests\Feature;

use App\ElementRepository;
use App\MenuRepository;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\App;
use Mtc\ContentManager\Contracts\ElementRepositoryContract;
use Mtc\ContentManager\Models\ContentElementField;
use Mtc\MercuryDataModels\ContentElement;
use Mtc\MercuryDataModels\Menu;
use Mtc\MercuryDataModels\MenuEntry;
use Tests\TenantTestCase;
use Tests\TestCase;

class MenuRepositoryTest extends TenantTestCase
{
    public function testExportToRemote()
    {
        $menu = Menu::factory()->create(['slug' => 'foo']);
        $menu2 = Menu::factory()->create();
        MenuEntry::factory()->create([
            'menu_id' => $menu->id,
        ]);

        $result = $this->repository()->exportToRemote([$menu->id, $menu2->id]);

        $this->assertCount(2, $result);
        $this->assertArrayHasKey('id', $result[0]);
        $this->assertArrayHasKey('slug', $result[0]);
        $this->assertArrayHasKey('title', $result[0]);
        $this->assertArrayHasKey('entries', $result[0]);

        $this->assertEquals(1, $result[0]['entries']->count());

    }

    public function testCanImport()
    {
        ContentElement::factory()->create(['slug' => 'foo']);
        $true = [
            'first_child_element_id' => null
        ];

        $true2 = [
            'first_child_element_id' => 'foo'        ];

        $false = [
            'first_child_element_id' => 'foobar',
        ];


        $this->assertTrue($this->repository()->canBeImported($true));
        $this->assertTrue($this->repository()->canBeImported($true2));
        $this->assertFalse($this->repository()->canBeImported($false));
    }

    public function testCheckImportValidity()
    {
        ContentElement::factory()->create(['slug' => 'baz-bar']);
        $entry = [
            'title' => 'foo faz far',
            'slug' => 'foo',
            'first_child_element_id' => null,
            'entries' => [
                [
                    'name' => 'lorem ipsum',
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(0, $result['errors']);

        $entry = [
            'title' => 'foo faz far',
            'slug' => '',
            'first_child_element_id' => 'baz-bar',
            'fields' => [
                [
                    'name' => 'lorem ipsum',
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(1, $result['errors']);

        $entry = [
             'title' => 'foo faz far',
            'slug' => 'foo',
            'first_child_element_id' => 'foobar',
            'entries' => [
                [
                    'name' => 'Foobar',
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(1, $result['errors']);
    }

    public function testImportRecord()
    {
        ContentElement::factory()->create(['slug' => 'foo-faz']);
        $data = [
            'title' => 'Foo',
            'slug' => 'foo',
            'first_child_element_id' => 'foo-faz',
            'entries' => [
                [
                    'id' => 1,
                    'name' => 'baz',
                    'slug' => 'baz',
                    'field_type' => 'text'
                ]
            ]
        ];
        $this->assertTrue($this->repository()->importRecord($data));
        $this->assertTrue(Menu::query()->where('slug', 'foo')->exists());
    }

    private function repository(): MenuRepository
    {
        return App::make(MenuRepository::class);
    }
}
