<?php

namespace Tests\Feature;

use App\ElementRepository;
use App\TemplateRepository;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\App;
use Mtc\ContentManager\Contracts\ElementRepositoryContract;
use Mtc\ContentManager\Models\ContentElementField;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Models\TemplateElement;
use Mtc\MercuryDataModels\ContentElement;
use Mtc\MercuryDataModels\GlobalContent;
use Tests\TenantTestCase;
use Tests\TestCase;

class TemplateRepositoryTest extends TenantTestCase
{
    public function testExportToRemote()
    {
        $element = ContentElement::factory()->create(['slug' => 'foo']);
        $template = Template::factory()->create();
        TemplateElement::factory()->create([
            'template_id' => $template->id,
            'element_id' => $element->id,
        ]);

        $result = $this->repository()->exportToRemote([$template->id]);

        $this->assertCount(1, $result);
        $this->assertArrayHasKey('id', $result[0]);
        $this->assertArrayHasKey('slug', $result[0]);
        $this->assertArrayHasKey('name', $result[0]);
        $this->assertArrayHasKey('elements', $result[0]);

        $field = $result[0]['elements'][0];
        $this->assertEquals('foo', $field['element_id']);

    }

    public function testCanImport()
    {
        ContentElement::factory()->create(['slug' => 'foo']);
        $true = [
            'elements' => []
        ];
        $true2 = [
            'elements' => [
                [
                    'element_id' => null,
                    'data' => [],
                ]
            ]
        ];

        $true3 = [
            'elements' => [
                [
                    'element_id' => 'foo',
                    'data' => [],
                ]
            ]
        ];

        $false = [
            'elements' => [
                [
                    'element_id' => 'foobar',
                    'data' => [],
                ]
            ]
        ];

        $false2 = [
            'elements' => [
                [
                    'element_id' => 'foo',
                    'data' => [],
                ],
                [
                    'element_id' => 'foobar',
                    'data' => [],
                ],
            ]
        ];

        $this->assertTrue($this->repository()->canBeImported($true));
        $this->assertTrue($this->repository()->canBeImported($true2));
        $this->assertTrue($this->repository()->canBeImported($true3));
        $this->assertFalse($this->repository()->canBeImported($false));
        $this->assertFalse($this->repository()->canBeImported($false2));
    }

    public function testCheckImportValidity()
    {
        ContentElement::factory()->create(['slug' => 'baz-bar']);
        Template::factory()->create(['slug' => 'baz-bar']);
        $entry = [
            'slug' => 'foo',
            'elements' => [
                [
                    'element_id' => null,
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(0, $result['errors']);

        $entry = [
            'slug' => '',
            'elements' => [
                [
                    'element_id' => null,
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(1, $result['errors']);
        $entry = [
            'slug' => 'baz-bar',
            'elements' => [
                [
                    'element_id' => null,
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(1, $result['errors']);

        $this->assertCount(1, $result['errors']);
        $entry = [
            'slug' => 'foo',
            'elements' => [
                [
                    'name' => 'Foobar',
                    'element_id' => 'foobar',
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(1, $result['errors']);
        $entry = [
            'slug' => 'foo',
            'elements' => [
                [
                    'name' => 'Foobar',
                    'global_content_id' => 'foobar',
                    'data' => [],
                ]
            ]
        ];
        $allData = [$entry];

        $result = $this->repository()->checkImportEntryValidity($entry, $allData);
        $this->assertCount(1, $result['errors']);
    }

    public function testImportRecord()
    {
        GlobalContent::factory()->create(['slug' => 'faz']);
        ContentElement::factory()->create(['slug' => 'foo']);
        $data = [
            'name' => 'Foo',
            'slug' => 'foo',
            'elements' => [
                [
                    'element_id' => 'foo',
                    'name' => 'baz',
                    'slug' => 'baz',
                    'field_type' => 'text'
                ],
                [
                    'global_content_id' => 'faz',
                    'name' => 'baz',
                    'slug' => 'baz',
                    'field_type' => 'text'
                ]
            ]
        ];
        $this->assertTrue($this->repository()->importRecord($data));
        $this->assertTrue(Template::query()->where('slug', 'foo')->exists());
    }

    private function repository(): TemplateRepository
    {
        return App::make(TemplateRepository::class);
    }
}
