<?php

namespace App\Traits;

use App\Filter\FilterIndex;
use Illuminate\Support\Collection;
use Mtc\ContentManager\Models\Page;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\Dealership;
use Mtc\MercuryDataModels\Form;
use Mtc\MercuryDataModels\Franchise;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\GlobalContent;
use Mtc\MercuryDataModels\Menu;
use Mtc\MercuryDataModels\Property;
use Mtc\MercuryDataModels\PropertyCategory;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleOffer;

trait LoadsDataFromSource
{
    private function getChoicesFromDataSource(?string $type): ?Collection
    {
        if (empty($type)) {
            return collect();
        }

        $slugs = collect();
        if (in_array($type, ['makes', 'fuel-types', 'body-styles'])) {
            // Not using relationship because of the master model morphing causing issues
            $slugs = FilterIndex::query()
                ->where('filter_type', $this->getIndexType($type))
                ->pluck('slug', 'filter_id');
        }
        return match ($type) {
            'dealerships' => Dealership::query()
                ->select(['id', 'name'])
                ->where('active', 1)
                ->orderBy('name')
                ->get()
                ->each(fn(Dealership $entry) => $entry->setHidden(['open_times', 'primaryDepartment'])),
            'franchises' => Franchise::query()
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get(),
            'makes' => VehicleMake::query()
                ->with('filterIndex')
                ->orderBy('name')
                ->get()
                ->map(fn ($entry) => [
                    'id' => $entry->id,
                    'name' => $entry->name,
                    'slug' => $slugs[$entry->id] ?? null,
                ]),
            'fuel-types' => FuelType::query()
                ->with('filterIndex')
                ->orderBy('name')
                ->get()
                ->map(fn ($entry) => [
                    'id' => $entry->id,
                    'name' => $entry->name,
                    'slug' => $slugs[$entry->id] ?? null,
                ]),
            'body-styles' => BodyStyleType::query()
                ->with('filterIndex')
                ->orderBy('name')
                ->get()
                ->map(fn ($entry) => [
                    'id' => $entry->id,
                    'name' => $entry->name,
                    'slug' => $slugs[$entry->id] ?? null,
                ]),
            'pages' => \Mtc\MercuryDataModels\Page::query()
                ->select(['id', 'title as name'])
                ->active()
                ->orderBy('name')
                ->get(),
            'menus' => Menu::query()
                ->select(['id', 'title as name'])
                ->active()
                ->orderBy('name')
                ->get(),
            'global-content' => GlobalContent::query()
                ->select(['id', 'name'])
                ->orderBy('name')
                ->get(),
            'forms' => Form::query()
                ->select(['id', 'name'])
                ->where('is_active', 1)
                ->orderBy('name')
                ->get(),
            'blog-pages' => Page::query()
                ->select(['id', 'title as name'])
                ->where('category', 'blog')
                ->orderBy('name')
                ->get(),
            'news-pages' => Page::query()
                ->select(['id', 'title as name'])
                ->where('category', 'news')
                ->orderBy('name')
                ->get(),
            'offers' => VehicleOffer::query()
                ->select(['id', 'name'])
                ->active()
                ->orderBy('name')
                ->get(),
            'vehicles' => Vehicle::query()
                ->select(['id', 'title', 'slug', 'registration_number'])
                ->active()
                ->orderBy('title')
                ->get()
                ->map(fn (Vehicle $vehicle) => [
                    'id' => $vehicle->id,
                    'name' => collect([
                        $vehicle->title,
                        $vehicle->registration_number,
                        $vehicle->slug,
                    ])->implode(' - '),
                ]),
            'taxonomies' => PropertyCategory::query()
                ->active()
                ->select(['id', 'name', 'slug as value'])
                ->get(),
            default => Property::query()
                ->whereHas('category', fn ($query) => $query->where('slug', $type))
                ->active()
                ->select(['name', 'slug as value'])
                ->get(),
        };
    }

    private function getIndexType(string $type): string
    {
        return match ($type) {
            'makes' => 'make',
            'fuel-types' => 'fuel_type',
            'body-styles' => 'body_type',
        };
    }
}
