<?php

namespace Tests\Feature;

use App\Crm\SendCopyViaMail;
use App\Facades\Settings;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Mockery\MockInterface;
use Mtc\Crm\Http\Requests\EnquiryTypeRequest;
use Mtc\Crm\Models\EnquiryAction;
use Mtc\Crm\Models\EnquiryType;
use Mtc\MercuryDataModels\Enquiry;
use Tests\TestCase;
use Tests\UserForTenant;

class EnquiryControllerTest extends TestCase
{
    use DatabaseMigrations;
    use UserForTenant;

    protected $tenancy = true;

    public function testRetry()
    {
        $this->partialMock(SendCopyViaMail::class, function (MockInterface $mock) {
            $mock->shouldReceive('handle')->andReturn(true);
        });

        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-copy-via-mail',
            'attempts' => 5,
            'processed_at' => null,
        ]);

        $this->assertEquals(5, EnquiryAction::query()->first()->attempts);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('enquiries.resubmit-action', [$enquiry, $action]));

        $response->assertStatus(200);
        $this->assertEquals(0, EnquiryAction::query()->first()->attempts);
    }

    public function testEnquiryTypeShowNoSalesforceData()
    {
        $enquiry_type = EnquiryType::factory()->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson('/api/crm/enquiry-types/' . $enquiry_type->id);

        $response->assertStatus(200);
        $this->assertArrayHasKey('salesforce_crm_enabled', $response->json());
        $this->assertArrayHasKey('salesforce_form_type', $response->json());
        $this->assertNull($response->json('salesforce_crm_enabled'));
        $this->assertNull($response->json('salesforce_form_type'));
    }

    public function testEnquiryTypeShowWithSalesforceData()
    {
        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'Salesforce',
            'name' => 'Enabled',
            'config_key' => 'salesforce-crm-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        $enquiry_type = EnquiryType::factory()->create([
            'data' => [
                'salesforce_form_type' => 'example',
            ]
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson('/api/crm/enquiry-types/' . $enquiry_type->id);

        $response->assertStatus(200);
        $this->assertArrayHasKey('salesforce_crm_enabled', $response->json());
        $this->assertArrayHasKey('salesforce_form_type', $response->json());
        $this->assertTrue($response->json('salesforce_crm_enabled'));
        $this->assertEquals('example', $response->json('salesforce_form_type'));
    }

    public function testEnquiryTypeUpdate()
    {
        Settings::make([
            'tab' => 'CRM',
            'section' => 'Integrations',
            'group' => 'Salesforce',
            'name' => 'Enabled',
            'config_key' => 'salesforce-crm-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        $request = new EnquiryTypeRequest();

        $request->merge([
            'name' => 'foo',
            'salesforce_form_type' => 'example',
        ]);

        $enquiry_type = EnquiryType::factory()->create([
            'data' => [
                'salesforce_form_type' => 'bar',
            ],
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson('/api/crm/enquiry-types/' . $enquiry_type->id, $request->input());

        $enquiry_type->refresh();

        $response->assertStatus(200);
        $this->assertEquals('example', $enquiry_type->data['salesforce_form_type']);
    }
}
