<?php

namespace App\Crm;

use App\Contracts\AddsDealershipFields;
use App\Contracts\ForwardsReservationToCrm;
use App\Facades\Settings;
use App\Services\EnquiryMaxApi;
use Illuminate\Support\Facades\App;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\VehicleReservations\Reservation;

class SendToEnquiryMax implements EnquiryAction, AddsDealershipFields, ForwardsReservationToCrm
{
    use CheckForExpiredAction;

    private EnquiryMaxApi $api;

    public function enabled(): bool
    {
        return Settings::get('crm-enquiry-max-enabled') === true;
    }

    /**
     * Data fields to add to form
     *
     * @return array
     */
    public function formAdditionalDataFields(): array
    {
        return [
            'fallback-dealership' => [
                'type' => 'text',
                'validation' => [
                    'required',
                ],
                'label' => 'Name of fallback dealership on EnquiryMax'
            ],
        ];
    }

    /**
     * Data fields to add to questions
     *
     * @return array[]
     */
    public function formQuestionDataFields(): array
    {
        return [
            'enquiry-max-field' => [
                'type' => 'select',
                'label' => 'Field mapped against EnquiryMax',
                'choices' => [
                    'leadType' => 'Lead Type',
                    "sourceOfEnquiry" => 'Source of Enquiry',
                    "methodOfContact" => 'Method of Contact',
                    "notes" => 'Notes',
                    "link" => 'Link',
                    "title" => 'Title',
                    "forename" => 'First Name',
                    "surname" => 'Last Name',
                    "companyName" => 'Company Name',
                    "bestTimeToCall" => 'Best Time To call',
                    "preferedContactMethod" => 'Preferred Contact Method',
                    "contactMethod" => 'Contact Method',
                    "homeTelephoneNumber" => 'Home Phone Number',
                    "mobileTelephoneNumber" => 'Mobile Phone Number',
                    "workTelephoneNumber" => 'Work Phone Number',
                    "emailAddress" => 'Email Address',
                    "address.line1" => 'Address Line 1',
                    "address.line2" => 'Address Line 2',
                    "address.line3" => 'Address Line 3',
                    "address.line4" => 'Address Line 4',
                    "address.postcode" => 'Address Postcode',
                    "marketing.post" => 'Mail marketing opt-in',
                    "marketing.phone" => 'Phone marketing opt-in',
                    "marketing.sms" => 'Sms marketing opt-in',
                    "marketing.email" => 'Email marketing opt-in',
                    'appointmentDateTime' => 'Appointment Date/Time',
                    'appointment.notes' => 'Appointment notes',
                ],
            ]
        ];
    }

    /**
     * Data fields to add to dealership
     *
     * @return array
     */
    public function dealershipAdditionalDataFields(): array
    {
        return [
            'enquiry-max-dealership' => [
                'type' => 'text',
                'label' => 'Name of dealership on EnquiryMax'
            ],
        ];
    }

    /**
     * Handle the action
     *
     * @param EnquiryActionModel $action
     * @param EnquiryModel $enquiry
     * @return bool
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        if ($this->expired($action)) {
            return null;
        }
        $this->api = App::make(EnquiryMaxApi::class);
        return $this->api->sendLead($enquiry, $action);
    }

    public function sendReservation(Reservation $reservation): bool
    {
        $this->api = App::make(EnquiryMaxApi::class);
        return $this->api->sendReservation($reservation);
    }

    /**
     * Message when successfully processed
     *
     * @return string
     */
    public function successDetails(): string
    {
        return __('crm.lead_registered_with_reference', [
            'reference' => $this->api->getResponseAttribute('leadId'),
        ]);
    }

    /**
     * Message when failed to process
     *
     * @return string
     */
    public function failureReason(): string
    {
        return $this->api->getResponseAttribute('message')
            ?? $this->api->getResponseAttribute('ErrorMessage')
            ?? '';
    }
}
