<?php

namespace Tests\Feature;

use App\Tier;
use Database\Seeders\Tenant\ChatSettingSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Mtc\MercuryDataModels\Setting;
use Tests\TestCase;
use Tests\UserForTenant;

class SettingControllerTest extends TestCase
{
    use RefreshDatabase;
    use UserForTenant;

    protected $tenancy = true;

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testShow()
    {
        Setting::factory(10)->create(['tab' => 'foo' ]);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.show', 'foo'));

        $response->assertStatus(200);

        $this->assertIsArray($response->json());
    }

    public function testIndex()
    {
        Setting::factory(10)->create(['tab' => 'foo' ]);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.index'));

        $response->assertStatus(200);

        $this->assertIsArray($response->json());
    }

    public function testStore()
    {
        tenant()->tier = Tier::ENTERPRISE->value;
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.store'), [
                'tab' => 'foo-baz',
                'section' => 'lorem',
                'group' => 'ipsum',
                'name' => 'dolor',
                'config_key' => 'foo',
                'value' => 'baz',
                'type' => 'string',
            ]);

        $response->assertStatus(200);
        $this->assertIsArray($response->json());
    }

    public function testUpdate()
    {
        $setting = Setting::factory()->create([
            'section' => 'foo',
            'config_key' => 'foo.baz.bar',
            'value' => 'foo-baz-bar',
        ]);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.settings.update', 'foo'), [
                'settings' => [
                    [
                        'config_key' => 'foo.baz.bar',
                        'value' => 'Lorem Ipsum'
                    ]
                ]
            ]);

        $response->assertStatus(200);

        $setting->refresh();
        $this->assertEquals('Lorem Ipsum', $setting->value);
    }

    public function testSettingsAreGroupedAndOrdered()
    {
        $this->seed(ChatSettingSeeder::class);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.show', 'CRM'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('LiveChat', $response->json('0.data'));
        $this->assertEquals('chat-live-chat-enabled', $response->json('0.data.LiveChat.0.config_key'));
    }
}
