<?php

namespace App\Http\Controllers;

use App\Exports\ChannelFeedExport;
use App\Exports\FacebookFeedExport;
use App\Exports\GoogleFeedExport;
use App\Exports\EasiChatFeedExport;
use App\Facades\Settings;
use App\Http\Resources\StormEnquiryListResource;
use App\Http\Resources\StormEnquiryResource;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\MercuryDataModels\Enquiry;
use Mtc\MercuryDataModels\Vehicle;
use Illuminate\Pagination\LengthAwarePaginator;
use Maatwebsite\Excel\Facades\Excel;
use Spatie\ArrayToXml\ArrayToXml;

class FeedController extends Controller
{
    public function channable(Request $request)
    {
        if ($request->input('api_token') != config('auth.api_auth_token')) {
            abort(401);
        }

        if (Settings::get('sales-channels-channel-feed-pagination-enabled', false)) {
            $export = new ChannelFeedExport();
            $vehicleQuery = $export->query();
            $perPage = $request->get('per_page', 15);
            $currentPage = $request->get('page', 1);
            $vehicles = $vehicleQuery->paginate($perPage, ['*'], 'page', $currentPage);
            $mappedVehicles = $vehicles->map(function ($vehicle) use ($export) {
                return $export->map($vehicle);
            });

            $paginatedData = new LengthAwarePaginator(
                $mappedVehicles,
                $vehicles->total(),
                $perPage,
                $currentPage,
                ['path' => $request->url(), 'query' => $request->query()]
            );

            return response()->json($paginatedData);
        } else {
            $tenant = tenant('id');
            return Storage::disk('file-storage')
                ->get("feeds/channel-feed/$tenant.csv");
        }
    }

    public function facebook(Request $request)
    {
        if ($request->input('api_token') != config('auth.api_auth_token')) {
            abort(401);
        }

        $export = new FacebookFeedExport();

        $fileName = 'facebook_feed.csv';
        if ($request->input('download') === 'true') {
            return Excel::download($export, $fileName, null, [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename=' . $fileName
            ]);
        }

        // Return CSV response to display in the browser
        $path = 'feeds/channel-feed/' . $fileName;
        $disk = config('app.env') == 'local' ? 'public' : 'file-storage';
        if (Excel::store($export, $path, $disk)) {
            return Storage::disk($disk)->get($path);
        }
    }

    public function google(Request $request, GoogleFeedExport $export)
    {
        if ($request->input('api_token') != config('auth.api_auth_token')) {
            abort(401);
        }

        if (!Settings::get('sales-channels-google-enabled', false)) {
            abort(404);
        }

        $fileName = 'google.csv';
        $disk = config('app.env') == 'local' ? 'public' : 'file-storage';

        if ($request->input('download') === 'true') {
            return Excel::download($export, $fileName, null, [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename=' . $fileName
            ]);
        }

        $path = 'feeds/google-feed/' . $fileName;
        if (Excel::store($export, $path, $disk)) {
            return Storage::disk($disk)->get($path);
        }
    }

    public function easichat(Request $request)
    {
        if ($request->input('api_token') != config('auth.api_auth_token')) {
            abort(401);
        }

        if (!Settings::get('sales-channels-easichat-enabled', false)) {
            abort(404);
        }

        $export = new EasiChatFeedExport();

        $fileName = 'easichat_stock.csv';
        if ($request->input('download') === 'true') {
            return Excel::download($export, $fileName, null, [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename=' . $fileName
            ]);
        }

        // Return CSV response to display in the browser
        $path = 'feeds/channel-feed/' . $fileName;
        $disk = config('app.env') == 'local' ? 'public' : 'file-storage';
        if (Excel::store($export, $path, $disk)) {
            return Storage::disk($disk)->get($path);
        }
    }

    public function stormEnquiries(Request $request)
    {
        $data = Enquiry::query()
            ->when(
                $request->filled('From'),
                fn($query) => $query->where('ingested_at', '>=', Carbon::parse($request->input('From')))
            )
            ->when(
                $request->filled('To'),
                fn($query) => $query->where('ingested_at', '<=', Carbon::parse($request->input('To')))
            )
            ->latest()
            ->paginate();
        return response(ArrayToXml::convert(
            (new StormEnquiryResource($data))->toArray($request),
            [
                'rootElementName' => 'EnquiryInformation',
                '_attributes' => [
                    "xmlns:i" => 'http://www.w3.org/2001/XMLSchema-instance',
                    "xmlns" => 'http://schemas.datacontract.org/2004/07/Storm.CRM.EntityDataModels',
                ]
            ],
        ))
            ->withHeaders([
                'Content-Type' => 'text/xml',
            ]);
    }
}
